<?php 
/**
 *
 * This Controller used to save, fetch or delete Magento Customers
 *
 * @category   Customers
 * @package    Magento API
 * @author     Tapas Ranjan<tapasranjanp@riaxe.com>
 * @copyright  2019-2020 Riaxe Systems
 * @license    http://www.php.net/license/3_0.txt  PHP License 3.0
 * @version    Release: @1.0
 */
namespace CustomerStoreSpace\Controllers;

use ComponentStoreSpace\Controllers\StoreComponent;

class StoreCustomersController extends StoreComponent
{
    /**
     * Get list of customer or a Single customer from the Magento API
     *
     * @author     tapasranjanp@riaxe.com
     * @date       18 Dec 2019
     * @parameter  Slim default params
     * @response   Array of list/one customer(s)
     */
    public function getCustomers($request, $response, $args)
    {
        $storeResponse = [];
        $getStoreDetails = get_store_details($request);
        try {
            if (isset($args['id']) && $args['id'] > 0) {
                // Fetching Single Customer details
                $filters = array(
                    'store' => $getStoreDetails['store_id'],
                    'customerId' => $args['id']
                );
                // $storeResponse = json_clean_decode($result, true);
                $result = $this->call("GET", "customer", "getStoreCustomerDetails", $filters);
                $storeResponse = json_clean_decode($result, true);
                // skip fields on condition
                $skip = $request->getQueryParam('skip');
                $skipFields = !empty($skip) ? explode(',', $skip) : [];
                if (in_array('shipping_address', $skipFields)) {
                    unset($storeResponse['shipping_address']);
                }
            } else {
                // Fetching all customer by filters
                $searchstring = (!empty($request->getQueryParam('name'))
                && $request->getQueryParam('name') != "")
                    ? $request->getQueryParam('name') : '';
                $page = (!empty($request->getQueryParam('page'))
                && $request->getQueryParam('page') != "")
                    ? $request->getQueryParam('page') : 0;
                $limit = (!empty($request->getQueryParam('perpage'))
                && $request->getQueryParam('perpage') != "")
                    ? $request->getQueryParam('perpage') : 20;
                $order = (!empty($request->getQueryParam('order'))
                && $request->getQueryParam('order') != "")
                    ? $request->getQueryParam('order') : 'asc';
                $orderby = (!empty($request->getQueryParam('orderby'))
                && $request->getQueryParam('orderby') != "")
                    ? $request->getQueryParam('orderby') : 'id';
                $type = (!empty($request->getQueryParam('type'))
                && $request->getQueryParam('type') != "")
                    ? $request->getQueryParam('type') : '';
                $fetch = (!empty($request->getQueryParam('fetch'))) ? $request->getQueryParam('fetch') : 'all';

                if ($type != '' && $type == 'quote') {
                    $limit = 100;
                }
                $filters = array(
                    'store' => $getStoreDetails['store_id'],
                    'searchstring' => $searchstring,
                    'page' => $page,
                    'limit' => $limit,
                    'order' => $order,
                    'orderby' => $orderby,
                    'customerNoOrder' => $request->getQueryParam('customer_no_order'),
                    'fromDate' => $request->getQueryParam('from_date'),
                    'toDate' => $request->getQueryParam('to_date'),
                    'fetch' => $fetch,
                );
                // $storeResponse = json_clean_decode($result, true);
                $result = $this->call("GET", "customer", "getStoreCustomers", $filters);
                $storeResponse = json_clean_decode($result, true);
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log('store', 'error', [
                'message' => $e->getMessage(),
                'extra' => [
                    'module' => 'Customer'
                ]
            ]);
        }
        return $storeResponse;
    }

    /**
     * GET: Total customer count
     *
     *
     * @author debashrib@riaxe.com
     * @date   04 Aug 2020
     * @return Integer
     */
    public function userCount() {
        $filters = array();
        try{
            $result = $this->call("GET", "customer", "getTotalCustomerCount", $filters);
            $resultData = json_clean_decode($result, true);
            $totalUser = $resultData['user_count'];
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:userCount',
                    ],
                ]
            );
        }
        return $totalUser;
    }

    /**
     * POST: Get all countries
     *
     *
     * @author debashrib@riaxe.com
     * @date   04 Aug 2020
     * @return Array
     */
    public function getAllCountries($request, $response) {
        $filters = array();
        try{
            $result = $this->call("GET", "customer", "getAllCountries", $filters);
            $resultData = json_clean_decode($result, true);

            $countries = $resultData['countries'];
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:getAllCountries',
                    ],
                ]
            );
        }
        return $countries;
    }

    /**
     * GET:Get all states by country code
     *
     * @param $country_code 
     *
     * @author debashrib@riaxe.com
     * @date   05 Aug 2020
     * @return Array
     */
    public function getAllStates($request, $response , $args) 
    {
        $filters = array(
            'countryCode' => $args['country_code']
        );
        try{
            $result = $this->call("GET", "customer", "getAllStatesByCode", $filters);
            $resultData = json_clean_decode($result, true);
            $states = $resultData['states'];
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:getAllStates',
                    ],
                ]
            );
        }
        return $states;
    }

    /**
     * Create Store Customer
     *
     * @author     debashrib@riaxe.com
     * @date       06 Aug 2020
     * @parameter  Slim default params
     * @response   Array
     */
    public function createCustomer($request, $response, $args)
    {
        $storeResponse = [];
        $getStoreDetails = get_store_details($request);
        $allPostPutVars = $request->getParsedBody();
        try {
            if (!empty($allPostPutVars)) {
                $filters = array(
                    'store' => $getStoreDetails['store_id'],
                    'data' => json_encode($allPostPutVars),
                );
                $result = $this->call("POST", "customer", "createCustomer", $filters);
                $resultData = json_clean_decode($result, true);
                if (!empty($resultData)) {
                    $storeResponse = [
                        'status' => $resultData['status'],
                        'message' => $resultData['message']
                    ];
                }
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer'
                    ]
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * Create Customer Shipping Address
     *
     * @author     debashrib@riaxe.com
     * @date       06 Aug 2020
     * @parameter  Slim default params
     * @response   Array
     */
    public function createShippingAddress($request, $response, $args)
    {
        $storeResponse = [];
        $getStoreDetails = get_store_details($request);
        $allPostPutVars = $request->getParsedBody();
        try {
            if (!empty($allPostPutVars)) {
                $filters = array(
                    'store' => $getStoreDetails['store_id'],
                    'data' => json_encode($allPostPutVars),
                );

                $result = $this->call("POST", "customer", "createShippingAddress", $filters);
                $resultData = json_clean_decode($result, true);

                if (!empty($resultData)) {
                    $storeResponse = [
                        'status' => $resultData['status'],
                        'message' => $resultData['message']
                    ];
                }
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer'
                    ]
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * Update Customer Shipping Address
     *
     * @author     debashrib@riaxe.com
     * @date       08 Aug 2020
     * @parameter  Slim default params
     * @response   Array
     */
    public function updateShippingAddress($request, $response, $args)
    {
        $storeResponse = [];
        $getStoreDetails = get_store_details($request);
        $allPostPutVars = $request->getParsedBody();
        try {
            if (!empty($allPostPutVars)) {
                $allPostPutVars['shipping_id'] = $args['id'];
                $filters = array(
                    'store' => $getStoreDetails['store_id'],
                    'data' => json_encode($allPostPutVars)
                );

                $result = $this->call("POST", "customer", "updateShippingAddress", $filters);
                $resultData = json_clean_decode($result, true);

                if (!empty($resultData)) {
                    $storeResponse = [
                        'status' => $resultData['status'],
                        'message' => $resultData['message']
                    ];
                }
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer'
                    ]
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * Delete Customer
     *
     * @author     debashrib@riaxe.com
     * @date       09 Aug 2020
     * @parameter  Slim default params
     * @response   Array
     */
    public function deleteCustomer($request, $response, $args)
    {
        $storeResponse = [];
        try {
            $getStoreDetails = get_store_details($request);
            if(isset($args['id']) && $args['id'] != '') {
                $customerIdArr = json_decode($args['id'], true);
                $customerIds = implode(',', $customerIdArr);
                $filters = array(
                    'store' => $getStoreDetails['store_id'],
                    'customerIds' =>  $customerIds
                );

                $result = $this->call("POST", "customer", "deleteCustomer", $filters);
                $storeResponse = json_clean_decode($result, true);

            }
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:deleteCustomer',
                    ],
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * GET:Minimal Customer details
     *
     * @author tapasranjanp@riaxe.com
     * @date   25 December 2020
     * @parameter  Slim default params
     * @response   Array
     */
    public function getQuoteCustomerDetails($customerId, $storeId, $shipId, $isAddress = false)
    {
        $storeResponse = [];
        try{
            $filters = array(
                'store' => (int)$storeId,
                'customerId' => intval(trim($customerId)),
                'shipId' =>  (int)$shipId,
                'isAddress' =>  (int)$isAddress
            );
            $result = $this->call("POST", "customer", "getStoreCustomerDetailsWithShipId", $filters);
            $result = json_clean_decode($result, true);
            if (!empty($result) && isset($result['id'])) {
                $storeResponse['customer'] = $result;
            }
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:getQuoteCustomerDetails',
                    ],
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * GET:User total count
     *
     ** @param $storeId
     * @author tapasranjanp@riaxe.com
     * @date   16 December 2020
     * @return int
     */
    public function getTotalStoreCustomer($storeId)
    {
        $storeResponse = [];
        try {
            // Fetching all Customer count
            $filters = array(
                'store' => (!empty($storeId)) ? $storeId : 1
            );
            $result = $this->call("POST", "customer", "getStoreCustomerCount", $filters);
            $storeResponse = json_clean_decode($result, true);
        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer count'
                    ]
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * Get All Customer's Id
     *
     * @author     tapasranjanp@riaxe.com
     * @date       13 Jan 2021
     * @parameter  Slim default params
     * @response   Array
     */
    public function getStoreCustomerId($request, $response, $args)
    {
        $storeResponse = [];
        try {
            $storeId = $request->getQueryParam('store_id') ? $request->getQueryParam('store_id') : 1;
            $page = $request->getQueryParam('page');
            $perPage = $request->getQueryParam('perpage');
            $filters = array(
                'store' => $storeId,
                'page' => $page,
                'perPage' =>  $perPage
            );
            $result = $this->call("POST", "customer", "getStoreCustomersId", $filters);
            $storeResponse = json_clean_decode($result, true);
        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Get Customers Ids'
                    ]
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * GET:Customer groups
     *
     * @author tapasranjanp@riaxe.com
     * @date   8th June
     * @return array
     */

    public function getAllStoreGroups($filters) {
        $storeResponse = [];
        try{
            if(isset($filters) && $filters != '') {
                $groupFilters = array(
                    'store' => $filters['store_id'],
                    'perPage' => $filters['perpage'],
                    'page' => $filters['page'],
                    'search' => $filters['search'],
                );
                $result = $this->call("POST", "customer", "getCustomerGroups", $groupFilters);
                $storeResponse = json_clean_decode($result, true);
            }
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:getAllStoreGroups',
                    ],
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * GET:Customer list in a customer groups
     *
     * @author tapasranjanp@riaxe.com
     * @date   8th June
     * @return array
     */
    public function getAllStoreGroupCustomers($groupId, $store_id, $returnCount = false)
    {
        $storeResponse = [];
        try{
            if(isset($groupId) && $groupId != '') {
                $filters = array(
                    'store' => (int)$store_id,
                    'groupId' =>  (int)$groupId,
                    'retrurnCount' =>  (int)$returnCount
                );
                $result = $this->call("POST", "customer", "getCustomersByGroupId", $filters);
                $storeResponse = json_clean_decode($result, true);
            }
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:getAllStoreGroupCustomers',
                    ],
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * GET:Customer Group Id
     *
     * @author tapasranjanp@riaxe.com
     * @date   8th June
     * @return array
     */
    public function getStoreGroupById($customerGroupId)
    {
        $storeResponse = [];
        try{
            if(isset($customerGroupId) && $customerGroupId != '') {
                $filters = array(
                    'store' => 1,//$store_id,
                    'groupId' =>  $customerGroupId
                );
                $result = $this->call("POST", "customer", "getGroupById", $filters);
                $storeResponse = json_clean_decode($result, true);
            }
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:getStoreGroupById',
                    ],
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * Update Customer groups
     *
     * @author tapasranjanp@riaxe.com
     * @date   8th June
     * @return array
     */
    public function updateStoreCustomerGroup($updateData){        
        $jsonResponse = [
            'status' => 0,
            'message' =>'Customer group not updated',
        ];
        try{
            if(isset($updateData) && $updateData != '') {
                $filters = array(
                    'groupId' => $updateData['id'],
                    'groupName' => $updateData['name'],
                    'customers' => json_encode($updateData['customer_id']),
                );
                $this->call("POST", "customer", "updateCustomerGroup", $filters);

                $jsonResponse = [
                    'status' => 1,
                    'message' => 'Customer group updated',
                ];
                
            }
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:updateStoreCustomerGroup',
                    ],
                ]
            );
        }
        return $jsonResponse;
    }

    /**
     * Create Customer groups
     *
     * @author tapasranjanp@riaxe.com
     * @date   8th June
     * @return array
     */
    public function createStoreCustomerGroup($name, $customers, $store_id){
        $jsonResponse = [];
        try{
            if(isset($name) && $name != '') {
                $filters = array(
                    'store' => $store_id,
                    'groupName' =>  $name,
                    'customers' =>  json_encode($customers)
                );
                $result = $this->call("POST", "customer", "createCustomerGroup", $filters);
                $storeResponse = json_clean_decode($result, true);
            }
            if(!empty($storeResponse)){
            $jsonResponse = [
                    'status' => 1,
                    'message' => 'Customer group created',
                    'data' => $storeResponse,
                    ];
            }
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:createStoreCustomerGroup',
                    ],
                ]
            );
        }
        return $jsonResponse;
    }

    /**
     * Delete Customer groups
     *
     * @author tapasranjanp@riaxe.com
     * @date   8th June
     * @return array
     */
    public function deleteStoreCustomerGroup($customerGroupId,$storeId){
        $storeResponse = [];
        try{
            if(isset($customerGroupId) && $customerGroupId != '') {
                $filters = array(
                    'store' => $storeId,
                    'groupId' =>  $customerGroupId,
                );
                $storeResponse = $this->call("POST", "customer", "deleteCustomerGroup", $filters);
                $storeResponse = json_clean_decode($storeResponse, true);
            }
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:deleteStoreCustomerGroup',
                    ],
                ]
            );
        }
        return $storeResponse;
    }

    /**
     * Get Customer group Id by Customer
     *
     * @author tapasranjanp@riaxe.com
     * @date   8th June
     * @return array
     */
    public function getStoreGroupIdByCustomerId($id){
        $storeResponse = [];
        try{
            if(isset($id) && $id != '') {
                $filters = array(
                    'store' => 1,//$storeId,
                    'customerId' =>  $id,
                );
                $storeResponse = $this->call("POST", "customer", "getCustomerGroupByCustomerId", $filters);
                $storeResponse = json_clean_decode($storeResponse, true);
            }
        }catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Customer:getStoreGroupIdByCustomerId',
                    ],
                ]
            );
        }
        return $storeResponse;
    }
}
