<?php
/**
 *
 * This Controller used to save, fetch or delete WooCommerce Products on various endpoints
 *
 * @category   Products
 * @package    WooCommerce API
 * @author     Original Author <tanmayap@riaxe.com>
 * @author     tanmayap@riaxe.com
 * @copyright  2019-2020 Riaxe Systems
 * @license    http://www.php.net/license/3_0.txt  PHP License 3.0
 * @version    Release: @1.0
 */
namespace ProductStoreSpace\Controllers;

use ComponentStoreSpace\Controllers\StoreComponent;

class StoreProductsController extends StoreComponent
{
    public function __construct()
    {
        parent::__construct();
    }
    /**
     * Get list of product from the Shopify API
     *
     * @author     steve@imprintnext.com
     * @date       12 dec 2021
     * @parameter  Slim default params
     * @response   Array of list/one product(s)
     */
    public function getProducts($request, $response, $args)
    {
        $productId = trim($args['id']);
        if (!empty($productId)) {
            return $this->getProductById($request, $response, $args);
            
        }
        $storeResponse = [];
        $products = [];
        

        $searchstring = $request->getQueryParam('name');
        $categoryid = (!empty($request->getQueryParam('category'))) ? $request->getQueryParam('category') : $args['categories'];
        $limit = $request->getQueryParam('per_page');
        $page = $request->getQueryParam('page');
        $orderby = $request->getQueryParam('orderby');
        $order = $request->getQueryParam('order');
        $listCatalogue = (!empty($request->getQueryParam('is_catalog'))? $request->getQueryParam('is_catalog') : 0);
        $fetchFilter = (!empty($request->getQueryParam('fetch'))? $request->getQueryParam('fetch') : 'limited');
        $isCustomize = $request->getQueryParam('is_customize'); //For Predeco Product

        // Get all requested Query params
        $filters = [
            'limit' => $limit,
            'page' => $page,
            'orderby' => $orderby,
            'order' => $order,
            'published_status' => 'published',
            'isCatalogue' => $listCatalogue,
            'getall' => $fetchFilter
        ];
        if ($categoryid && $categoryid != '') {
            $filters['collection_id'] = $categoryid;
        } else {
            $filters['collection_id'] = '';
        }
        if ($searchstring && $searchstring != '') {
            $filters['title'] = $searchstring;
        }
        
        try {
            $getProducts = $this->getAllProducts($filters, $isCustomize);
            $getProducts = json_clean_decode($getProducts);
            $getTotalProductsCount = $getProducts['meta']['pagination']['total'];
            $getProducts = $getProducts['data'];            
            foreach ($getProducts as $key => $product) {
                //print_r($product['images']); exit;
                $productImages = [];             
                if (isset($product['images']) && count($product['images']) > 0) {
                    foreach ($product['images'] as $prodImgKey => $prodImg) {
                        if($prodImg['is_thumbnail'] == 1 && $prodImgKey > 0)
                        {
                            $productImages[$prodImgKey] = $productImages[0];
                            $productImages[0] = $prodImg['url_thumbnail'];
                        }
                        else {
                            $productImages[$prodImgKey] = $prodImg['url_thumbnail'];
                        }                        
                        
                    }
                }
                //Choosing variant price as product price
                $productPrice = 10000000;
                foreach ($product['variants'] as $pv) {
                    if (!empty($productId)) {
                        //For product details
                        if ($pv['price'] < $productPrice) {
                            $productPrice = array(
                                'current' => $pv['price'],
                                'regular_price' => $pv['price'],
                                'sale_price' => $pv['price'],
                            );
                            break;
                        }
                    } else {
                        if ($pv['calculated_price'] < $productPrice) {
                            $productPrice = $pv['calculated_price'];
                            break;
                        }
                        if ($pv['inventory_policy'] =="continue" || $pv['inventory_quantity'] > 0 || !empty($pv['inventory_quantity'])) {
                            break; //Product is unsold
                        }
                    }
                }                
                $productType = empty($product['base_variant_id']) ? "variable" : "simple";
                $stock = 0;
                foreach ($product['variants'] as $var) {
                    if ($var['inventory_level'] > 0) {
                        $stock = $var['inventory_level'];
                        break;
                    }
                }
                $products[$key] = [
                    'id' => $product['id'],
                    'variation_id' => ($productType == 'variable') ? $product['variants'][0]['id'] : $product['id'],
                    'name' => $product['name'],
                    'stock' => $stock,
                    'type' => $productType,
                    'is_sold_out' => ($stock > 0?false:true),
                    'sku' => (count($product['variants']) > 0) ? $product['variants'][0]['sku'] : '',
                    'is_decoration_exists' => 0,
                    'print_profile' => [],
                    'price' => $productPrice,
                    'image' => $productImages,
                ];
                if ($fetchFilter == 'all') {
                    $showIndesigner = $this->getShowInDesignerId();
                    $predecoId = $this->checkCreateCollection('predeco_products');
                    $preDecoMeta = $this->getMetafieldDetails($product['id']);
                    $products[$key]['custom_design_id'] = $preDecoMeta;
                    $products[$key]['is_redesign'] = (in_array($showIndesigner,$product['categories'])) ? 1:0;
                    $products[$key]['is_decorated_product'] = (in_array($predecoId,$product['categories'])) ? 1:0;
                }
            }
            if (isset($products) && is_array($products) && !empty($products)) {
                $storeResponse = [
                    'status' => 1,
                    'total_records' => $getTotalProductsCount,
                    'products' => $products,
                ];
            } else {
                $storeResponse = [
                    'status' => 0,
                    'total_records' => $getTotalProductsCount,
                    'message' => 'No products available',
                    'products' => [],
                ];
            }

        } catch (\Exception $e) {
            $storeResponse = [
                'status' => 0,
                'message' => 'Invalid request',
                'exception' => $e->getMessage(),
            ];
        }
        // Reset Total product Count
        $_SESSION['productsCount'] = 0;
        return $storeResponse;
    }

    /**
     * Get product details from the Shopify API
     *
     * @author     steve@imprintnext.com
     * @date       12 dec 2021
     * @parameter  Slim default params
     * @response   Array of list/one product(s)
     */
    public function getProductById($request, $response, $args)
    {
        try {
            $storeResponse = $this->getShopProductDetails($args['id']);
            //echo json_encode($product); exit;
        } catch (\Exception $e) {
            $storeResponse = [
                'status' => 0,
                'message' => 'Invalid request',
                'exception' => $e->getMessage(),
            ];
        }

        // Reset Total product Count
        $_SESSION['productsCount'] = 0;
        return $storeResponse;
    }

    /**
     * Get list of category/subcategory or a Single category/subcategory from the Shopify store
     *
     * @author     sonali@imprintnext.com
     * @date       19 jan 2022
     * @parameter  Slim default params
     * @response   Array of list/one categories/category
     */
    public function getCategories($request, $response, $args)
    {
        
        return $this->getCollections();
        
    }

     /**
     * Get: Get minimal product details from the store end
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument parameters
     *
     * @author steve@imprintnext.com
     * @date   28 feb 2022
     * @return Array of list/one product(s)
     */
    public function getProductShortDetails($request, $response, $args)
    {
        $storeResponse = [];

        $productId = to_int($args['product_id']);
        $variantId = to_int($args['variant_id']);
        if ($productId > 0 && $variantId > 0) {
            try {
                $productInfo = array(
                    'variantID' => $variantId,
                    'productID' => $productId,
                );
                $storeResponse = $this->getVariantShortInfo($productInfo);
            } catch (\Exception $e) {
                $storeResponse['exception'] = show_exception() === true
                ? $e->getMessage() : '';
            }
        }

        return $storeResponse;
    }

    /**
     * Get list of Color Variants from the WooCommerce API as per the product
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Arguments
     *
     * @author sonali@imprintnext.com
     * @date   27 Dec 2021
     * @return Json
     */
    public function colorsByProduct($request, $response, $args)
    {
        $storeResponse = [
            'status' => 0,
            'records' => 0,
            'message' => 'No Color available',
            'data' => [],
        ];
        $filters = [
            'product_id' => $args['product_id'],
            'attribute' => $args['slug'],
            'color_type' => $args['color'],
        ];
        $options = [];
        foreach ($filters as $filterKey => $filterValue) {
            if (isset($filterValue) && $filterValue != "") {
                $options += [$filterKey => $filterValue];
            }
        }
        try {
            $storeResponse = $this->getColorOptions($options);
        } catch (\Exception $e) {
            $storeResponse['exception'] = show_exception() === true
            ? $e->getMessage() : '';
        }

        return $storeResponse;
    }
    /**
     * Get: Get all Attributes List from Big-Commerce store API
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author divya@imprintnext.com
     * @date   19th January 2022
     * @return Array list of Attributes
     */
    public function getOnlyAttribute($request, $response)
    {
        return $this->getStoreProductAttrs();
    }
    /**
     * Get: Get all Attributes List from Store-end
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author steve@imprintnext.com
     * @date   17 Feb 2019
     * @return Array list of Attributes
     */
    public function storeAttributeList($request, $response)
    {
        $storeResponse = [];
        $productId = $request->getQueryParam('product_id');
        if (!empty($productId)) {
            $attributeList = [];
            $getProductDetail = $this->getAttributeByProductId($productId);
            if (!empty($getProductDetail)) {
                foreach ($getProductDetail as $attribute) {
                    $attributeList[] = [
                        'id' => $attribute['id'],
                        'name' => $attribute['name'],
                        'terms' => $attribute['options'],
                    ];
                }

                $storeResponse = $attributeList;
            }
        } else {
            try {
                $getAllAttributes = $this->getAttributeByProductId();
                if (!empty($getAllAttributes)) {
                    $storeResponse = $getAllAttributes;
                }
            } catch (\Exception $e) {
                $storeResponse = [];
                create_log(
                    'store', 'error',
                    [
                        'message' => $e->getMessage(),
                        'extra' => [
                            'module' => 'Create Variations',
                        ],
                    ]
                );
            }
        }

        return $storeResponse;
    }
     /**
     * Get: get tier Details of a product
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author sonali@imprintnext.com
     * @date   22 Jan 2022
     * @return Array records
     */
    public function productTierDiscounts($request, $response, $args)
    {
         $variantId = $request->getQueryParam('variant_id');
         return $this->getvarientTierDiscounts($args['productID'],$variantId);
       
    }
    /**
     * Get: Get Attribute List for Variants with Multiple attribute
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Argument
     *
     * @author sonali@imprintnext.com
     * @date   10th June 2020
     * @return Array records and server status
     */
    public function storeMultiAttributeVariantDetails($request, $response, $args)
    {
        $storeResponse = [];
        $productId = to_int($args['pid']);
        $variationId = to_int($args['vid']);
        $attribute = $args['attribute'];
        $isPrice = (!empty($args['price'])) ? $args['price'] : (!empty($request->getQueryParam('price')) ? $request->getQueryParam('price') : 0);
        $option['per_page'] = 100;
        try {
            $storeData = array("productID" => $productId, "variantID" => $variationId, "option" => $attribute);
            $storeData['is_price'] = $isPrice;
            $finalArray = $this->getVariantCombination($storeData,$args);
            if (!empty($finalArray)) {
                $storeResponse = $finalArray;
            }
        } catch (\Exception $e) {
            // Store exception in logs
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Store variant attribute details with quantity',
                    ],
                ]
            );
        }
     return $storeResponse;
    }
    /**
     * Post: Save predecorated products into the store
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author steve@imprintnext.com
     * @date   11 feb 2022
     * @return Array records and server status
     */
    public function saveProduct($request, $response)
    {
        $storeResponse = $productImages = [];
        $getPostData = (isset($saveType) && $saveType == 'update')
        ? $this->parsePut() : $request->getParsedBody();

        if (!empty($getPostData['data'])) {
            try{
                $predecorData = json_clean_decode($getPostData['data'], true);

                $colorProductImageUrl = [];
                if (isset($predecorData['ref_id']) && $predecorData['ref_id'] > 0) {
                    $refId = $predecorData['ref_id'];
                    $templateJsonPath = path('abs', 'design_state') . 'templates/' . $refId . '.json';
                    if (file_exists($templateJsonPath)) {
                        $tempData = file_get_contents($templateJsonPath);
                        $tempData = json_clean_decode($tempData, true);
                        $colorProductImageUrl = $tempData['color_product_image_url'];
                    }
                }
                $predecorData['color_images'] = $colorProductImageUrl;
                if (is_array ($predecorData['product_image_url']) && empty($predecorData['product_image_url'])) {
                    $productImages[] = $tempData['design_product_data'][0]['design_urls'][0];
                    $predecorData['product_image_url'] = $productImages;
                  }
                  if (is_array ($predecorData['product_image_url']) && !empty($predecorData['product_image_url'])) {
                    $uploadedFileNameList = do_upload(
                        'product_image_files',
                        path('abs', 'product'),
                        [],
                        'array'
                    );
                    foreach ($uploadedFileNameList as $uploadedImage) {
                        $productImages[] = path('read', 'product') . $uploadedImage;
                    }
                    $predecorData['product_image_url'] = !empty($productImages) ? $productImages : $predecorData['product_image_url'];
                  }
                  
                foreach ($predecorData['attributes'] as $attrKey => $attributes) {
                    foreach ($attributes['attribute_options'] as $optKey => $option) {
                        $predecorData['attributes'][$attrKey]['attribute_options'][$optKey] = addslashes($option);
                    }
                }
                $storeResponse = $this->createDecoratedProduct($predecorData);
            }catch(\Exception $e){
                create_log(
                    'store', 'error',
                    [
                        'message' => $e->getMessage(),
                        'extra' => [
                            'module' => 'post pre decorated data',
                            'controller' => 'storeproductcontroller',
                        ],
                    ]
                );

            }
            
        }
        return $storeResponse;
    }
    /**
     * Post: Validate SKU or Name at Magento store
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author steve@imprintnext.com
     * @date   10 Feb 2022
     * @return Int product id if exist
     */
    public function validateStoreSkuName($request, $response)
    {
        
        $getStoreDetails = get_store_details($request);
        $allPostPutVars = (isset($saveType) && $saveType == 'update')
        ? $this->parsePut() : $request->getParsedBody();
        $filters = array(
            'sku' => isset($allPostPutVars['sku']) ? $allPostPutVars['sku'] : null,
            'name' => isset($allPostPutVars['name']) ? $allPostPutVars['name'] : null,
            'store' => $getStoreDetails['store_id'],
        );
        return $this->SkuNameValidate($filters, $isCustomize);
    }

     /**
     * Bigcommerce REST API call helper function
     *
     * @param $productId Carte id & line item Id
     *
     * @author steve@imprintnext.com
     * @date   15 jan 2022
     * @return return Product Meta Data
     */
    public function getStoreProductMetaDetails($productId){
        return $this->getMetafieldDetails($productId);
    }

    /**
     * Bigcommerce store products variant id
     *
     * @param $productId Carte id & line item Id
     *
     * @author divya@imprintnext.com
     * @date   20 jan 2022
     * @return return Product variant Id
     */
    public function variantOptions($request, $response, $args){
        $productId = to_int($args['pid']);
        $optionValue = "size=small#color=Red";
        $finalArray = $this->getOnlyVariants($productId,$optionValue);
            if (!empty($finalArray)) {
                $storeResponse = $finalArray;
            }
        return $storeResponse;
    }

    /**
     * Get: check product stock 
     *
     * @param $productId
     * @param $variantId
     * @param $storeId
     *
     * @author steve@imprintnext.com
     * @date   30 dec 2021
     * @return Json records
     */
    public function checkProductStock($productId,$variantId,$storeId = 1)
    {   
        return $this->getProductStock($productId,$variantId);
    }
    /**
     * Get: product add to store
     *
     * @param $productId
     * @param $variantId
     * @param $storeId
     *
     * @author sonali@imprintnext.com
     * @date   09 Mar 2022
     * @return Json records
     */
    public function addProductToStore($productData, $catalog, $priceData, $storeID)
    {
        $productArr = [];
        try{
            if (!empty($productData)) {
                foreach ($productData as $k => $v) {
                    $price = $v['price'];
                    $storeCategories = $v['categories'];
                    $catalog_price = $v['catalog_price'] ? $v['catalog_price'] : $price;
                    $params = array('style_id' => $v['style_id'], "catalog_code" => $catalog);
                    $returnData = api_call_by_curl($params, 'product');
                    $predecorData = $returnData['data'];
                    $predecorData['old_product_id'] = $v['old_product_id'];
                    $sideName = $predecorData['variations'][0]['side_name'];
                    if (!empty($predecorData)) {
                        $productId = $this->addCatalogProductToStore($priceData, $predecorData, $price, $catalog_price, $storeCategories);
                        if ($productId) {
                            $productArr[$k]['product_id'] = $productId;
                            $productArr[$k]['product_side'] = $sideName;
                            $productArr[$k]['style_id'] = $v['style_id'];
                        }
                    }
                }
            }

        }catch(\Exception $e){
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'import the catalog',
                        'controller' => 'storeproductcontroller',
                    ],
                ]
            );

        }
        
        return $productArr;
    }
    
    /**
     * Get: product add to store in CSV format
     *
     *
     * @author sonali@imprintnext.com
     * @date   14 Mar 2022
     * @return Json records
     */

    public function createProductImportCSV($request, $response, $args){
        $predecoDetails = $request->getParsedBody();
        $productData = json_clean_decode($predecoDetails['product_data']);
        $catalog = $predecoDetails['catalog_code'];
        $assetsPath = ASSETS_PATH_W;
        $cataloAssetsPath = $assetsPath . 'catalog';
        if (!is_dir($cataloAssetsPath)) {
            mkdir($cataloAssetsPath, 0755);
        }
        $headerData = [
            "Item Type" , "Product Code/SKU" , "Product Name" ,"Category" ,"Product Image File-1" ,
            "Price" ,"Product Weight" ,"Track Inventory","Option Set" ,"Product Description" ,"Current Stock Level","gtin"
        ];
        
        $randNo = getRandom();
        $csvFilename = $randNo . '.csv';
        if (!empty($productData)) {
            $productArray = [];
            $productArray[0] = $headerData;
            $i = 1;
            foreach ($productData as $v) {
                $price = $v['price'];
                $catalog_price = $v['catalog_price'];
                $params = array("catalog_code" => $catalog, 'style_id' => $v['style_id']);
                $returnData = api_call_by_curl($params, 'product');
                $productData = $returnData['data'];
                $arraySize = $productData['size_data'];
                $arrayColor = $productData['color_data'];
                $color = '';
                if (!empty($arrayColor)) {
                    foreach ($arrayColor as $cl) {
                        $color .= $cl . ', ';
                    }
                }
                $size = '';
                if (!empty($arraySize)) {
                    foreach ($arraySize as $sz) {
                        $size .= $sz . ', ';
                    }
                    // $sizes = rtrim($size, ', ');
                }
                $productImageUrl = $productData['images']['src'];
                $productArray[] = [
                    "Product" , $productData['sku'] , $productData['name'] ,"/show_in_designer" ,$productImageUrl,
                    $productData['price'] ,"","none" ,"" ,$productData['description'],135,""
                ];
                if (!empty($productData['variations'])) {
                    $j = 0;
                    if (count($productData['variations']) > 1) {
                        foreach ($productData['variations'] as $keys => $variations) {
                            if ($keys > 0) {
                                $quantity = $variations['quantity'];
                                if ($variations['piece_price'] > 0) {
                                    $diffPrice = $price - $catalog_price;
                                    $variantPrice = $variations['piece_price'] + $diffPrice;
                                } else {
                                    $variantPrice = $variations['piece_price'];
                                }
                                $image_path = $variations['image_path'];
                                $attribue = $variations['attributes'];
                                $gtin = !empty($variations['gtin']) ? $variations['gtin'] : "";
                                if(!empty($attribue)){
                                    $color = $variations['attributes']['color'];
                                    $size = $variations['attributes']['size'];
                                }
                                $productArray[] = [
                                    "SKU" , "[S]Size=$size,[C]Color=$color" , "" ,"" ,$image_path[0] ,
                                    $variantPrice,$variations['unit_weight'] ,"" ,"" ,"",$quantity,$gtin
                                ];
                                $j++;
                            }
                        }
                    }
                }
                $i++;
                $newArr = $productArray;
                if (!empty($newArr)) {
                    $cfilename = $cataloAssetsPath . '/' . $csvFilename;
                    if (is_dir($cataloAssetsPath)) {
                        $fp = fopen($cfilename, 'w');
                        foreach ($newArr as $fields) {
                            fputcsv($fp, $fields);
                        }
                    }
                    fclose($fp);
                }
            }
        }
        return $csvFilename;
    }

    /**
     * Get: get variants of a product
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author steve@imprintnext.com
     * @date   16 March 2022
     * @return Array records
     */
    public function productVariants($productId, $storeId = 1)
    {
        $result = [
            'status' => 0, 'variants' => []
        ];
        if (!empty($productId)) {
            $variants = $this->getStoreVariants($productId);
            if (!empty($variants)) {
                $result = [
                    'status' => 1, 'variants' => $variants
                ];
            }
        }
        return $result;
    }

    /**
     * Get: Get Product Description by Product Id
     *
     * @param $productId  Product Id
     *
     * @author steve@imprintnext.com
     * @date  17 March 2022
     * @return string
     */
    public function getProductDescription($productId, $storeId = 1){        
        $product = $this->getShopProductDetails($productId);
        return $product['products']['description'];
    }

    /**
     * Get: Product Attribute Pricing Details by Product Id from Bigcommerce store
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     * @param $args     Slim's Arguments
     *
     * @author steve@riaxe.com
     * @date   17 March 2022
     * @return All store attributes
     */
    public function storeProductAttrPrc($request, $response, $args)
    {
        $productId = trim($args['id']);
        return $this->getAttributeByProductId($productId);
        
    }

     /**
     * Get: Update products stocks, price for selected product variants
     *
     * @param $productStockArr  (productId, variantId, stock, price)
     *
     * @author divya@imprintnext.com
     * @date   25 April 2022
     * @return Array (status = 1/0)
     */
    public function stockPriceUpdate($variantStocks = [])
    {
        return $this->variantStockUpdate($variantStocks);
    }

    /**
     * Get total product count from the BigCommerce API
     *
     * @author divya@imprintnext.com
     * @date   20 May 2022
     * @return count
     */
    public function totalProductCount(){
        $getCountDetails = $this->getTotalProductsCount();
        return array("total" => $getCountDetails);
    }

    /**
     * Internal: Saves product API data in cache files 
     * for Bigcommerce API call limitation.
     * @author sonali@imprintnext.com
     * @date   11 jan 2023
     * @return Array records and server status
     */
    public function saveProductAPIasCache($productID)
    {
        $productDIR = BIGCOMMERCE_CACHE_FOLDER . $productID;
        if (!is_dir($productDIR)) {
            mkdir($productDIR);
        }
        $variantsDIR = $productDIR . "/variants/";
        if (!is_dir($variantsDIR)) {
            mkdir($variantsDIR);
        }
        $productData = $this->getBigcommerceProductInfo($productID);
        $productImage = $this->getProductImageInfo($productID);
        if (!empty($productData)) {
            $thisProdCacheFile = $productDIR . '/' . $productID . ".json";
            file_put_contents($thisProdCacheFile, $productData);
            $imageCacheFile = $productDIR . '/' . "image.json";
            file_put_contents($imageCacheFile, $productImage);
        }
        $productVariant = json_decode($productData, true);
        $productVariant = $productVariant['data']['variants'];
        foreach ($productVariant as $variant) {
            $getSingleVariant = $this->getProductSingleVariantInfo($productID, $variant['id']);
            $variantCacheFile = $variantsDIR . $variant['id'] . ".json";
            file_put_contents($variantCacheFile, $getSingleVariant);
        }
        return $productData;
    }

    /* getting current store information */
    public function getStoreDatas() {
        return $this->getStoreInformation();
    }
}
