<?php
/**
 * This is a common store component which will be used across all Module-store-controllers
 */

namespace ComponentStoreSpace\Controllers;

use App\Components\Controllers\Component;
use App\Components\Models\BigcommerceCart;
use App\Components\Models\quote_items;


class StoreComponent extends Component {
    public $shopDomain;
    private $token;
    private $apiKey;
    private $secret;
    private $lastResponseHeaders = null;
    public $thumbSize = "small";
    public $fullSize = "";

    function __construct(){
        if(!isset($_SESSION)) {
            session_start();
        }
        parent::__construct();      
        /* BIGCOMMERCE SETTINGS */
        $this->accessToken = BIG_ACCESSTOKEN;
        $this->apiPath = BIG_API_PATH;
        $this->clientId = CLIENTID;
        $this->clientSecret = CLIENTSECRET;
         /* BIGCOMMERCE SETTINGS */

    }

    /**
     * Bigcommerce REST API call helper function
     *
     * @param $httpMethod  REST API call method (GET, POST, PUT, DELETE)
     * @param $path  Bigcommerce API end point
     * @param $params  Bigcommerce API parameters if any
     * @param $sessionKey  Session identifying key for pagination.
     *
     * @author steve@imprintnext.com
     * @date   12 jan 2022
     * @return Bigcommerce REST API call response in array
     */
    public function call($httpMethod , $path, $params = array(), $version = 'v3', $fetchRaw="cache"  )
    {   
        $this->apiPath = str_replace("v3",$version,BIG_API_PATH);
        switch ($httpMethod) {
            case "GET":
            $result = $this->get($path, $params , $fetchRaw);
            break;
            case "POST":
            $result = $this->post($path, $params , $fetchRaw="cache");
            break;
            case "PUT":
            $result = $this->put($path, $params , $fetchRaw="cache");
            break;
            case "DELETE":
            $result = $this->delete($path, $params , $fetchRaw="cache");        
            break;       
            default:
            echo "Error";
        }
        return $result;        
    }
    
    /**
     * Performs a get request to the instantiated class
     * 
     * Accepts the apiUrl to perform the request on
     * 
     * @param $apiUrl string $apiUrl a string to perform get on
     * @return results or var_dump error
     */
    private function get($path, $params = array(), $fetchRaw = "cache"){
        try{
            if($fetchRaw == "cache" ) {
                $response = $this->getAPIDataFromCache($path);
            }
            if(empty($response)){
                $headers = array(
                    'x-auth-token:'.$this->accessToken,
                    'Accept: application/json',
                    'Content-Type: application/json'
                );
                $completeApiPath = $this->apiPath.$path  ;    
                if(!empty($params))
                {
                    $pathValue ='';
                    foreach($params as $key => $value)
                    {
                        $pathValue .= $key .'='. $value."&" ; 
                    }
                    $pathValue = rtrim($pathValue, '&');
                    $completeApiPath = $this->apiPath.$path."?".$pathValue;
                }
                $curl = curl_init();
                
                curl_setopt_array($curl, array(
                  CURLOPT_URL => $completeApiPath,
                  CURLOPT_RETURNTRANSFER => true,
                  CURLOPT_ENCODING => "",
                  CURLOPT_MAXREDIRS => 10,
                  CURLOPT_TIMEOUT => 30,
                  CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                  CURLOPT_CUSTOMREQUEST => 'GET',
                  CURLOPT_HTTPHEADER => $headers,
                ));
                $response = curl_exec($curl);
            }
            $jsonDecode = json_decode($response, true);
            $errorMsg = $jsonDecode['errors'];
            $http_status = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            if ($http_status >= 400) {
                if (isset($errorMsg)) {
                    throw new \Exception(json_encode($errorMsg, true));
                } else {
                    throw new \Exception($jsonDecode[0]['message']);
                }
            }
            curl_error($curl);
            curl_close($curl);
        }catch(\Exception $e){
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Issue in GET method',
                        'controller' => 'storecomponent',
                    ],
                ]
            );
            
        }        
        
        return $response;
    }
    /**
     * get data from cache folder
     * 
     * 
     * @param $path
     * @return get the data from cache file.
     */
    private function getAPIDataFromCache($path){
        $APIContent = array();
        if (strpos($path, 'catalog/products/') !== false && strpos($path, '/images') == false && strpos($path, '/variants') == false) {
            $thisProductID = substr($path, strrpos($path, '/') + 1);
            $productDIR = BIGCOMMERCE_CACHE_FOLDER . $thisProductID;
            $thisProdCacheDIR = $productDIR . '/' . $thisProductID . ".json";
            if (!empty($thisProductID) && file_exists($thisProdCacheDIR)) {
                $cacheContent = file_get_contents($thisProdCacheDIR);
            }
            $APIContent = $cacheContent;
        }elseif(strpos($path, 'catalog/products/') !== false && strpos($path, '/images') !== false){
            $APIQry = substr($path, strrpos($path, '/') - 3);
            $thisProductID = substr($APIQry, 0, strpos($APIQry, '/'));
            $productDIR = BIGCOMMERCE_CACHE_FOLDER . $thisProductID.'/image.json';
            if (!empty($thisProductID) && file_exists($productDIR)) {
                $cacheContent = file_get_contents($productDIR);
            }
            $APIContent = $cacheContent;
           
        }elseif(strpos($path, 'catalog/products/') !== false && strpos($path, '/variants') !== false){
            $thisProductID = substr($path, strrpos($path, '/') - 12);
            $thisProductID = substr($thisProductID, 0, strpos($thisProductID, '/'));
            $APIQry = substr($path, strrpos($path, '/'));
            $variantId = substr($APIQry,1);
            $variantDIR = BIGCOMMERCE_CACHE_FOLDER . $thisProductID . '/variants/' . $variantId . '.json';
            if (!empty($thisProductID) && !empty($variantId) && file_exists($variantDIR)) {
                $cacheContent = file_get_contents($variantDIR);
            }
            $APIContent = $cacheContent;
        }else{
            $APIContent = "";
        }
        return $APIContent;
    }
    /**
     * Performs a post request to the instantiated class
     * 
     * Accepts the url to perform the request on, and fields to be sent
     * 
     * @param string $url a string to perform get on
     * @param array $fields an array to be sent in the request
     * @return results or var_dump error
     */
    private function post($path, $params = array(), $fetchRaw = "cache"){
        try{
            $headers = array(
                'x-auth-token:'.$this->accessToken,
                'Accept: application/json',
                'Content-Type: application/json'
            );         
            $completeApiPath = $this->apiPath.$path;
            if(!empty($params['include']))
            {
                $completeApiPath = $completeApiPath."?"."include=".$params['include'];
                unset($params['include']);
            }
            if($params['module'] == 'cart') {
                unset($params['module']);
                $jsonParam = json_encode(["line_items" => $params]);
            }
            else{
                $jsonParam = json_encode($params);
            }
            global $error;
            $curl = curl_init();
            curl_setopt($curl, CURLOPT_URL, $completeApiPath);
            curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl, CURLOPT_VERBOSE, 1);
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, "POST");
            curl_setopt($curl, CURLOPT_POSTFIELDS, $jsonParam);
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
            $response = curl_exec ($curl);
            $jsonDecode = json_decode($response,true);
            $errorMsg = $jsonDecode['errors'];
            $http_status = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            if($http_status >= 400){
                if(isset($errorMsg)){
                    throw new \Exception(json_encode($errorMsg,true));
                }else{
                    throw new \Exception($jsonDecode[0]['message']);
                }
            }
            curl_getinfo($curl, CURLINFO_HTTP_CODE);
            $header_size = curl_getinfo($curl, CURLINFO_HEADER_SIZE);
            $body = substr($response, $header_size);
            curl_close ($curl);
            if ($http_status == 200) {
                return json_decode($response, true);
            } else {
                return $this->error($body, $url, $json, 'PUT');
            }
        }catch(\Exception $e){
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Missing required fields in POST method',
                        'controller' => 'storecomponent',
                    ],
                ]
            );
            
        }
        return $response;
    }
    /**
     * Performs a put request to the instantiated class
     * 
     * Accepts the url to perform the request on, and fields to be sent
     * 
     * @param string $url a string to perform get on
     * @param array $fields an array to be sent in the request
     * @return results or var_dump error
     */
    private function put($resource, $fields, $version= "v3") {
        try{
            // $this->apiPath = str_replace("v3", $version, $this->apiPath);
            $APIURL = $this->apiPath . $resource;
            $headers = array(
                'x-auth-token:' . $this->accessToken,
                'Accept: application/json',
                'Content-Type: application/json'
            );
            $json = json_encode($fields);
            $curl = curl_init();
            curl_setopt($curl, CURLOPT_URL, $APIURL);
            curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl, CURLOPT_VERBOSE, 1);
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, "PUT");
            curl_setopt($curl, CURLOPT_POSTFIELDS, $json);
            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
            $response = curl_exec($curl);
            $http_status = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            $header_size = curl_getinfo($curl, CURLINFO_HEADER_SIZE);
            $body = substr($response, $header_size);
            curl_close($curl);
            if ($http_status == 200) {
                return json_decode($response, true);
            } else {
                return $this->error($body, $url, $json, 'PUT');
            }
        }catch(\Exception $e){
            create_log(
                'store', 'error',
                [
                    'message' => $e->getMessage(),
                    'extra' => [
                        'module' => 'Missing required fields in PUT method',
                        'controller' => 'storecomponent',
                    ],
                ]
            );

        }
        

    }
    /**
     * Performs a delete request to the instantiated class
     * 
     * Accepts the url to perform the request on
     * 
     * @param string $url a string to perform get on
     * @return proper response or var_dump error
     */
    private function delete($path, $params = array(), $sessionKey = "", $fetchRaw = false) {
        $headers = array(
            'x-auth-token:'.$this->accessToken,
            'Accept: application/json',
            'Content-Type: application/json'
            );
        $completeApiPath = $this->apiPath.$path;
        $pathValue ='';
        if(!empty($params))
        {
            $pathValue ='';
            foreach($params as $key => $value)
            {
                $pathValue .= $key .'='. $value."&" ; 
            }
            $pathValue = rtrim($pathValue, '&');
            $completeApiPath = $this->apiPath.$path."?".$pathValue;
        }
        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $completeApiPath);
        curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_VERBOSE, 1);
        curl_setopt($curl, CURLOPT_CUSTOMREQUEST, "DELETE");
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
        $response = curl_exec($curl);
        $http_status = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $header_size = curl_getinfo($curl, CURLINFO_HEADER_SIZE);
        $headers = substr($response, 0, $header_size);
        $body = substr($response, $header_size);
        curl_close ($curl);
        if ($http_status == 204) {
            return $http_status . 'DELETED';
         } else {
            return $this->error($body, $url, null, 'DELETE');
         }
    }

    /**
     * Internal function to get Show in designer collection id
     *
     * @author steve@imprintnext.com
     * @date 12 Dec 2022
     *
     * @return Array of  catagories
     */
    public function getShowInDesignerId()
    {
        $params = [
            'name:like' => 'show_in_designer'
        ];
        $response = $this->call('GET', 'catalog/categories', $params);
        $data = json_clean_decode($response, true);
        return $data['data'][0]['id'];
    }

    /**
     * Internal function to get Show in designer collection id
     *
     * @author steve@imprintnext.com
     * @date 15 Dec 2022
     * 
     * @return encoded string
     */
    private function arrayUtf8Encoder($data)
    {
        if (is_string($data)) {
        return utf8_encode($data);
        }
        if (!is_array($data)) {
            return $data;
        } 
        $result = array();
        foreach ($data as $i => $d){
            $result[$i] = self::arrayUtf8Encoder($d);
        }   
        return $result;
    }

    /** ************************* Product Module Start **************************** **/

     /**
     * GET: Fetching all products from store
     * 
     * @author steve@imprintnext.com
     * @date 16 Jan 2022
     * @param filter parameters
     * @return Array of all product
     */
    public function getAllProducts($filters, $isCustomize = '') {
        $order = !empty($filters['order']) ? $filters['order']  : 'Desc';
        if(!empty($filters['orderby'])){            
            $productParams['sort'] = 'name';
            $productParams['direction'] = $order;
        }
        if(empty($filters['orderby'])){
            $productParams['sort'] = 'date_modified';
            $productParams['direction'] = $order;
        }
        if(!empty($filters['title'])){
            $productParams['keyword'] = urlencode($filters['title']);
        }
        $order = ($filters['order'] == "desc")? SORT_DESC : SORT_ASC;
        $productParams['name:like'] = empty($filters['name'])? '' : $filters['name'];
        $productParams['page'] = empty($filters['page'])? 1 : $filters['page'];
        $productParams['limit'] = empty($filters['limit'])? 100 : $filters['limit'];
        $categoryid = $filters['collection_id'];
        $productParams['include']  = 'images,custom_fields,variants' ;
        
        if ($filters['is_catalog'] == 0 && $isCustomize == 0 && $filters['getall'] != 'all') {
            $productParams['categories:not_in'] = $this->checkCreateCollection('predeco_products');
            $productParams['categories:in'] = $this->getShowInDesignerId();
        }
        if ($categoryid && $categoryid != '') {
            $productParams['categories:in'] = $categoryid;
        }
        if($filters['isCatalogue'] == 1){
            $productParams['categories:in'] = $this->checkCreateCollection('catlog_products');
        }
        if($isCustomize == 1 ){
            $productParams['categories:in'] = $this->checkCreateCollection('predeco_products');
        }
        $products = $this->call('GET', 'catalog/products', $productParams);
        return $products;
    }

    /**
     * GET: will fetch available attributes from store
     * 
     * @author divya@imprintnext.com
     * @date 19th Jan 2021
     * @param filter parameters
     * @return Array of all product's attribute
     */
    public function getStoreProductAttrs() {
        $productParams['page'] = 1;
        $productParams['limit'] = 25;
        $productParams['include_fields'] = "ID";
        $allAttrs = array();
        $productParams['categories:in'] = $this->getShowInDesignerId();
        //call product api to get list of products
        $products = $this->call('GET', 'catalog/products', $productParams);
        $productArray = json_decode($products, true);
        $productIDs = array_column($productArray['data'], "id");
        $uniqueAttrs = array();
        // get each products attribute list
        foreach($productIDs as $productID) {
            $thisProdAttrs = $this->call('GET', 'catalog/products/'.$productID.'/options');
            $attrData = json_clean_decode($thisProdAttrs, true);
            if(!empty($attrData['data'])){
                foreach($attrData['data'] as $attrValue){
                    $thisAttribute = array("id" => strtolower($attrValue['display_name']), "name" => $attrValue['display_name']); 
                    if (!in_array(strtolower($attrValue['display_name']), $uniqueAttrs)) {
                        $uniqueAttrs[] = strtolower($attrValue['display_name']);
                        $allAttrs[] =  $thisAttribute;
                    }
                }
            }
            $attrData =  $allAttrs; 
         }
         return $attrData;        
    }

     /**
     * GET:  will get available variant combinations/options of a product and the available inventory
     * 
     * @author Sonali@imprintnext.com
     * @date 26 Apr 2021
     * @param filter parameters
     * @return Array of all product's attribute
     */
    public function getVariantCombination($storeData,$args){
        $productDetails = $this->call('GET', 'catalog/products/'.$args['pid'].'/options');
        $productDetails = json_decode($productDetails ,true);
        $productDetails = $productDetails['data'];
        $variantDetails = $this->call('GET', 'catalog/products/'.$args['pid'].'/variants');
        $variantDetails = json_decode($variantDetails ,true);
        $variantDetails = $variantDetails['data'];
        $singleVar = $this->call('GET', 'catalog/products/'.$args['pid'].'/variants/'.$args['vid']);
        $singleVar = json_decode($singleVar ,true);
        $singleVar = $singleVar['data'];
        $tierDetails = $this->getvarientTierDiscounts($args['pid'],$args['vid']);
        $attribute = [];
        $requireAttPos = 0;
        $attrKey = 0;
        foreach($productDetails as $product){
            if (!in_array($product['display_name'], $attribute)) {
                $attribute[$attrKey] = $product['display_name'];
                $attrKey++;
            }
        }

        for ($pos = 0; $pos < $attrKey; $pos++) {
            if (strtolower($singleVar['option_values'][$pos]['option_display_name']) != strtolower($storeData['option'])) {
                $variantAttValue[] = [
                    "label"  => $singleVar['option_values'][$pos]['label'],
                    "option_display_name" => $singleVar['option_values'][$pos]['option_display_name'],

                ];
            }
        }
    
        $uniqueSizes = [];
        foreach($variantDetails as $variant){
            $checkVar = $this->checkVariantAttribute ($variant['option_values'], $variantAttValue);
            if($checkVar){
                $count = 0;
                foreach($variant['option_values'] as $optionValue){
                    $count = count($variant['option_values']);
                    if(strtolower($storeData['option']) == strtolower($optionValue['option_display_name']) && !in_array($optionValue['label'],$uniqueSizes)){
                       $uniqueSizes[] = $optionValue['label'];
                       $optionDetails = array();
                       $optionDetails['id'] = $optionValue['id'];
                       $optionDetails['name'] = $optionValue['label'];
                       $optionDetails['variant_id'] = $variant['id'];
                       $attribute = [];
                        foreach ($variant['option_values'] as $optionColor) {
                            $key = strtolower($optionColor['option_display_name']);
                            $attribute[] = [
                                $key. '_id' => $optionColor['id'],
                                $key => $optionColor['label']
                            ];
                        }
                        $attributeDetails = call_user_func_array('array_merge', $attribute);
                       
                        if($storeData['is_price'] > 0 || $count == $attrKey){
                           $optionDetails['inventory'] = [
                               "stock" => $variant['inventory_level'],
                               "min_quantity" => '1' ,
                               "max_quantity" => '1000',
                               "quantity_increments" => '1' 
                           ];
                           $optionDetails['attributes'] = !empty($attributeDetails) ? $attributeDetails : [];
                           $optionDetails['price'] = $variant['calculated_price'];
                           $optionDetails['tier_prices'] = !empty($tierDetails) ? $tierDetails : [];
                        }
                        $options[$storeData['option']][] = $optionDetails;
                    }
                }
                
            }
            
        }
        return $options;
    }
    /**
     * Get: Get all Attributes List from Store-end
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author Debashish@imprintnext.com
     * @date   26 Apr 2021
     * @return Array list of Attributes
     */
    private function checkVariantAttribute ($attributeArr, $checkArr){
        $variantMatched = false;
        $flagChecks = array();
        foreach ($checkArr as $attrVal){
            foreach ($attributeArr as $varAttr){
                if (count(array_intersect($varAttr, $attrVal)) == 2){
                    $flagChecks[] = array_intersect($varAttr, $attrVal);
                    break 1;
                }
            }

        }

        if (count((array)$checkArr) == count($flagChecks)) {
            $variantMatched = true;
        }
       
        return $variantMatched;
    }

    /**
     * Get: Get all Attributes List from Store-end
     *
     * @param $request  Slim's Request object
     * @param $response Slim's Response object
     *
     * @author steve@imprintnext.com
     * @date   12 Dec 2021
     * @return Array list of Attributes
     */
    public function getAttributeByProductId($productId)
    {
        $productAttrJson = $this->call('GET', 'catalog/products/'.$productId."/options");
        $productAttr = json_clean_decode($productAttrJson, true);
        $productAttr = $productAttr['data'];
        $attributes = [];
        foreach ($productAttr as $value) {
            $attributeValues = array();
            $thisAttribute['id'] = $value['id'];
            $thisAttribute['name'] = $value['display_name'];
            foreach ($value['option_values'] as $optionKey => $optionValue) {
                $attributeValues[$optionKey] = [
                        'id' => $optionValue['label'],
                        'name' => $optionValue['label'],
                    ];
            }
            $thisAttribute['options'] = $attributeValues;
            $attributes[] = $thisAttribute;
        }
        return $attributes;
    }

    /**
     * geting product details from store
     * 
     * @author steve@imprintnext.com
     * @date   12 Dec 2021
     * @param product id
     * @return product details
     */
    public function getShopProductDetails($productId) {
        $productParams['include']  = 'images,custom_fields,variants';
        $product = $this->call('GET', 'catalog/products/'.$productId,$productParams);
        $product = json_clean_decode($product, true);
        $product = $product['data'];                
        foreach ($product['images'] as $pi) {
            $images[$pi['id']] = $pi;
        }
        // Get Product Attributes
        $attributes = $this->getAttributeByProductId($productId);
        foreach ($attributes as $value) { 
            if(strtolower($value['name']) == 'color'){
                $color_look = $value['options'][0]['name'];
            }
            if(strtolower($value['name']) == 'size'){
                $size_look = $value['options'][0]['name'];
            }           
        }
        //Choosing variant price as product price
        foreach ($product['variants'] as $pv) {
            if ($pv['price'] != '') {
                $productPrice = $pv['price'];
                break;
            }else {
                $productPrice = $pv['calculated_price'];
            }
        }
        $productType = empty($product['base_variant_id']) ? "variable" : "simple";
        $isDecorationExists = "";
        $printProfiles = "";
        $products = [
            'id' => $product['id'],
            'variant_id' => $product['variants'][0]['id'],
            'name' => $product['name'],
            'description' => $product['description'],
            'type' => $productType,
            'sku' => (count($product['variants'])>0) ? $product['variants'][0]['sku'] : '',
            'is_decoration_exists' => $isDecorationExists,
            'print_profile' => $printProfiles,
            'price' => $productPrice,
        ];        
        // get variant images
        $productImages = [];
        $productDefaultImage = array_column($product['images'], 'is_thumbnail');
        $productKey = array_search( 1,$productDefaultImage);
        $productImages = $this->getVariantSideImages($product['images'],$color_look,$size_look,$product['images'][$productKey]['url_zoom']);
        $products['images'] = $productImages;
        $products['slug'] = $product['name'];
        foreach ($product['categories'] as $catValue) {
            $catJsonData = $this->call('GET', 'catalog/categories/'.$catValue);
            $catData = json_clean_decode($catJsonData, true);
            $catData = $catData['data'];
            $categories[] = [
                'id' => $catData['id'],
                'name' => $catData['name'],
            ];
        }
        $products['categories'] = $categories; 
        $products['attributes'] = $attributes; 
        $products['date_created'] = date('Y-m-d', strtotime($product['date_created'])); 
        if(isset($products) && is_array($products) && !empty($products)) {
            $storeResponse = [
                'products' => $products
            ];
        } else {
            $storeResponse = [
                'status' => 0,
                'message' => 'No products available',
                'data' => []
            ];
        }
        return $storeResponse;
    }

    /**
     * Getting Collections/attributes from store
     * 
     * @author sonali@imprintnext.com
     * @date 19 jan 2022
     * @param arraynull
     * @return array of collections
     */
    public function getCollections() {
        $checkCategorie = array();
        $products = $this->call('GET', 'catalog/categories',"product");
        $productArr = json_decode($products ,true);
        $this->arrayUtf8Encoder($productArr);
        foreach($productArr['data'] as $productCategorie){
            if ($productCategorie['name'] != "show_in_designer" && $productCategorie['name'] != "Shop All" && 
            $productCategorie['name'] != "predeco_products" && $productCategorie['name'] != "catlog_products"){
                $thisCategory = array(
                    'id' => $productCategorie['id'],
                    'name' => $productCategorie['name'],
                    'parent_id' => $productCategorie['parent_id']
                );
                $checkCategorie[] = $thisCategory;
            }
        }
        
        return $checkCategorie;
    }

    /**
     * Retrieve a single product's category(not used currently)
     *
     * @author steve@imprintnext.com
     * @date 18 Dec 2019
     * @param array|object Request parameters
     * @return string New product id
     */
    public function createDecoratedProduct($params) {

        if(!empty($params)){
            try{
                $metaDataInfo = array();
                if ($params['is_redesign'] > 0) {
                    $categories[] = $this->getShowInDesignerId();
                }
                $categoryName = "predeco_products";
                $createCatagory = $this->createCollection($categoryName);
                $categories[] = $createCatagory[2];
                if (!empty($params['categories'])) {
                    foreach ($params['categories'] as $category) {
                        $categories[] = $category['category_id'];
                    }
                }

                $colorImages = $params['color_images'] ? $params['color_images'] : [];
                $names = array_column($colorImages, 'name');


                $categories = array_unique($categories);
                // get parent product images
                $imgArr = array();
                foreach ($params['product_image_url'] as $key => $img) {
                    $key = $key + 1;
                    $imgArr[] = array("image_url" => $img, "is_thumbnail" => true, "sort_order" => $key);
                }
                $lastSortKey = sizeof($imgArr)-1;
                //Combine Variant Details with attribute combination
                $variantArr = array();
                if (!empty($params['attributes'])) {
                    foreach ($params['attributes'] as $key => $attribute) {
                        $key += 1;
                        $attributes[] = strtolower($attribute['attribute_name']);
                        $inputOptions[] = $attribute['attribute_options'];
                    }
                    $variantCombinations = $this->variantCombinations($inputOptions);
                    foreach ($variantCombinations as $combKey => $optionData) {
                        $variantPredecoImage = [];

                        $optionInfo = [];
                        foreach ($optionData as $key => $optionVal) {
                            $optionInfo[] = array('label' => $optionVal, 'option_display_name' => $attributes[$key]);
                            if(array_search($optionVal , $names) !== false){
                                $row_index = array_search($optionVal , $names);
                                $variantPredecoImage = $colorImages[$row_index]['sides'];
                            }
                        }
                        $variantImage = '';
                        if(!empty($variantPredecoImage)){
                            $variantImage = $variantPredecoImage[0];
                        }else{
                            $variantImage = $imgArr[0]['image_url'];
                        }
                        
                        $combKey++;
                        $thisVariant = array(
                            "option_values" => !empty($optionInfo) ? $optionInfo : array(array('label' => $optionData, 'option_display_name' => $attributes[0])),
                            "sku" => $params['sku'] . "_" . $combKey,
                            "inventory_level" => $params['quantity'],
                            "inventory_tracking " => 100,
                            "price" => $params['price'],
                            "categories" => $categories,
                            "image_url" => $variantImage,
                        );
                        $variantArr[] = $thisVariant;
                        foreach ($variantPredecoImage as $img) {
                            $lastSortKey = $lastSortKey+1;
                            $imgArr[] = array("image_url" => $img, "is_thumbnail" => 0, "sort_order" => $lastSortKey);
                        }
                    }
                }


                $productArray = array(
                    "name" => $this->bigCommerce_body(addslashes($params['name'])),
                    "weight" => 100,
                    "description" => $this->bigCommerce_body(addslashes($params['description'])),
                    "is_visible" => true,
                    "type" => "physical",
                    "sku" => $params['sku'],
                    "inventory_level" => $params['quantity'],
                    "inventory_tracking " => 100,
                    "price" => $params['price'],
                    "categories" => $categories,
                    "images" => array_reverse($imgArr),
                );
                if (!empty($optionInfo) && !empty($variantArr)) {
                    $productArray['variants'] = $variantArr;
                }
                if (empty($optionInfo)) {
                    $productArray['variants'] = $variantArr;
                }
                $metaDataInfo = array(
                    "permission_set" => "app_only",
                    "namespace" => "imprint_design_id",
                    "key" => "imprint_design_id",
                    "value" => "reff_" . $params['ref_id'],
                    "description" => strval($params['parent_product_id'])
                );
                $filters['sku'] = $params['sku'];
                $filters['name'] = $params['name'];
                $isCustomize = 1;
    
                $storeResponse = $this->SkuNameValidate($filters, $isCustomize);
                if ($storeResponse > 0) {
                    $status = 0;
                }
                $productArray['module'] = 'product';
                $newProduct = $this->call('POST', 'catalog/products', $productArray);
                $newProduct = json_decode($newProduct, true);
                $newProduct =  $newProduct['data'];
                $productId = $newProduct['id'];
                if (!empty($metaDataInfo)) {
                    $this->call('POST', 'catalog/products/' . $productId . '/metafields', $metaDataInfo);
                }
            }catch (\Exception $e) {
                $storeResponse = [
                    'status' => 0,
                    'message' => 'Invalid request',
                    'exception' => $e->getMessage(),
                ];
            }
            return $storeResponse = [
                'product_id' => $newProduct['id'],
                'status' => $status,
            ];

        }else{
            throw new \Exception("Post all the required pre record data");
        }
        
        
    }

    /**
     * GET: Variant information by ID
     * 
     * @author steve@imprintnext.com
     * @date 10 Jan 2022
     * @param $variant variant ID 
     * @return array New product details
     */
    public function getVariantShortInfo($productInfo) {
        //getting product meta details info of the product(in case of predeco product only)
        $metaDetails = $this->call('GET', 'catalog/products/'.$productInfo['productID'].'/metafields','v3');
        $metaDetailsInfo = json_clean_decode($metaDetails, true);
        $parentId = intval($metaDetailsInfo['data'][0]['description']);
        $metaDetaImage = $this->call('GET', 'catalog/products/'.$parentId.'/images','v3');
        $metaDataImageInfo = json_clean_decode($metaDetaImage, true);
        $attributes = $this->getAttributeByProductId($parentId);
        foreach ($attributes as $value) { 
            if(strtolower($value['name']) == 'color'){
                $color_look = $value['options'][0]['name'];
            }
            if(strtolower($value['name']) == 'size'){
                $size_look = $value['options'][0]['name'];
            }           
        }
        //getting default images of the parent product
        $parentProductImages = [];
        if(!empty($metaDataImageInfo['data'])){
            $parentProdDefaultImages = array_column($metaDataImageInfo['data'], 'is_thumbnail');
            $ParentProdKey = array_search( 1,$parentProdDefaultImages);
        }
        $parentProductImages = $this->getVariantSideImages($metaDataImageInfo['data'],$color_look,$size_look,$metaDataImageInfo['data'][$ParentProdKey]['url_zoom']);
        $productCategory = $this->checkCreateCollection('predeco_products');

        $variantID = $productInfo['variantID'];
        $productId = $productInfo['productID'];
        $productParams['include']  = 'images,custom_fields,variants' ;
        $product = $this->call('GET', 'catalog/products/'.$productId,$productParams);
        $product = json_clean_decode($product, true);
        $product = $product['data'];        
        if ($variantID == $productId) {
            $variantID = $product['variants'][0]['id'];
        }
        $variant = $this->call('GET', 'catalog/products/'.$productId.'/variants/'. $variantID);
        $variant = json_clean_decode($variant, true);
        $variant = $variant['data'];
        $attributes = [];        
        // for tier price
        foreach ($product['images'] as $pi) {
            $images[$pi['id']] = $pi;
        }
        $variantDetails = array();
        if (!empty($variant)) {
            $variantDetails['name'] = $product['name'];
            $variantDetails['price'] = $variant['calculated_price'];         
            $color_look = '';
            $size_look = '';
            foreach ($variant['option_values'] as $value) {
                if(strtolower($value['option_display_name']) == 'color'){
                    $color_look = $value['label'];
                    $attributes[strtolower($value['option_display_name'])] = 
                     ['id' => $value['id'] , 'name' => $value['label'] ,'attribute_id' => $value['option_id'] ];
                }elseif(strtolower($value['option_display_name']) == 'size'){
                    $attributes[strtolower($value['option_display_name'])] = 
                     ['id' => $value['id'] , 'name' => $value['label'] ,'attribute_id' => $value['option_id'] ];
                    $size_look = $value['label'];
                }else{
                    $attributes[strtolower($value['option_display_name'])] = 
                    ['id' => $value['id'] , 'name' => $value['label'] ,'attribute_id' => $value['option_id'] ];

                }
            }
            $variantDetails['tier_prices'] = $this->getvarientTierDiscounts($productId,$variantID);
            $variantDetails['attributes'] = $attributes;
            foreach ($product['categories'] as $catValue) {
                $catJsonData = $this->call('GET', 'catalog/categories/'.$catValue);
                $catData = json_clean_decode($catJsonData, true);
                $catData = $catData['data'];
                $categories[] = [
                    'id' => $catData['id'],
                    'name' => $catData['name'],
                ];
            }
            $variantDetails['categories'] = $categories;
            $thisProdType = (strpos(strtolower($variant['title']), 'default title') !== false ? "simple" : "configurable");
            if ($variantID == $productId) {
                $thisProdType = "simple";
            }            
            $color_lookup = ($thisProdType == 'simple' ? '_' : $color_look);
            $variantDetails['color_lookup'] = $color_lookup;            
            $variantDefaultImage = $variant['image_url'] ;
            if($variantDefaultImage == '')
            {
                $productDefaultImage = array_column($product['images'], 'is_thumbnail');
                $productKey = array_search( 1,$productDefaultImage);
                $variantDefaultImage = $product['images'][$productKey]['url_zoom'];
            }
            $variantDetails['images'] = (!in_array($productCategory, $product['categories'])) ? $this->getVariantSideImages($images,$color_lookup,$size_look,$variantDefaultImage) : $parentProductImages;
                  
        }
        return $variantDetails;
    }

    /**
     * GET: Variant side image
     * 
     * @author steve@imprintnext.com
     * @date 10 Jan 2022
     * @param $haystack
     * @param $needles
     */
    private function str_contains_all($haystack, array $needles) {
        foreach ($needles as $needle) {
            if (strpos($haystack, $needle) === false) {
                return false;
            }
        }
        return true;
    }
    
    /**
     * GET: Variant side image
     * 
     * @author steve@imprintnext.com
     * @date 10 Jan 2022
     * @param $images
     * @param $color_lookup
     * @param $size_look
     * @param $variantDefaultImage
     * @return array image array
     */
    private function getVariantSideImages($images,$color_lookup,$size_look,$variantDefaultImage){        
        $thisImage = array();
        $i = 0;
        foreach ($images as $pimg) {
            if ($this->str_contains_all(strtolower($pimg['url_zoom']), array($color_lookup, "front"))) {                    
                $thisImage[$i]['labels'] = "Front";
                $thisImage[$i]['src'] = $pimg['url_zoom'];
                $thisImage[$i]['thumbnail'] = $pimg['url_tiny'];
                $i++;
            } elseif ($this->str_contains_all(strtolower($pimg['url_zoom']), array($color_lookup, "left"))) {
                $thisImage[$i]['labels'] = "Left";
                $thisImage[$i]['src'] = $pimg['url_zoom'];
                $thisImage[$i]['thumbnail'] = $pimg['url_tiny'];
                $i++;
            } elseif ($this->str_contains_all(strtolower($pimg['url_zoom']), array($color_lookup, "right"))) {
                $thisImage[$i]['labels'] = "Right";
                $thisImage[$i]['src'] = $pimg['url_zoom'];
                $thisImage[$i]['thumbnail'] = $pimg['url_tiny'];
                $i++;
            } elseif ($this->str_contains_all(strtolower($pimg['url_zoom']), array($color_lookup, "back"))) {
                $thisImage[$i]['labels'] = "Back";
                $thisImage[$i]['src'] = $pimg['url_zoom'];
                $thisImage[$i]['thumbnail'] = $pimg['url_tiny'];
                $i++;
            } elseif ($this->str_contains_all(strtolower($pimg['url_zoom']), array($color_lookup, "top"))) {
                $thisImage[$i]['labels'] = "Top";
                $thisImage[$i]['src'] = $pimg['url_zoom'];
                $thisImage[$i]['thumbnail'] = $pimg['url_tiny'];
                $i++;
            } else {
                if ($this->str_contains_all(strtolower($pimg['url_zoom']), array($color_lookup, "bottom"))) {
                    $thisImage[$i]['labels'] = "Bottom";
                    $thisImage[$i]['src'] = $pimg['url_zoom'];
                    $thisImage[$i]['thumbnail'] = $pimg['thumbnail'];
                    $i++;
                }
            }
            if (empty($thisImage)) {
                $thisImage[$i]['labels'] = "Front";
                $thisImage[$i]['src'] = $variantDefaultImage;
                $thisImage[$i]['thumbnail'] = $variantDefaultImage;
            }
        }
        return $thisImage;
    }
    
    /**
     * GET: product color options
     * 
     * @author sonali@imprintnext.com
     * @date 05 July 2020
     * @param $filter variant information
     * @return array colors details
     */
    public function getColorOptions($filter)
    {
        $checkOption = array();
        $uniqueColors = array();
        $productParams['include']  = 'images,custom_fields,variants';
        $productDetails = $this->call('GET', 'catalog/products/' . $filter['product_id'], $productParams);
        $productDetails = json_decode($productDetails, true);
        $productAttributeDetails = $this->call('GET', 'catalog/products/'. $filter['product_id'].'/options');
        $productAttributeDetails = json_decode($productAttributeDetails, true);
        $defaultImage = $productDetails['data']['images'];
        $productVariant = $productDetails['data']['variants'];
        foreach ($productVariant as $singleVar) {
            $tierDetails = $this->getvarientTierDiscounts($filter['product_id'], $singleVar['id']);
            $imageUrl = $singleVar['image_url'];
            if (!isset($imageUrl) || $imageUrl == '') {
                $productDefaultImage = array_column($defaultImage, 'is_thumbnail');
                $productKey = array_search(1, $productDefaultImage);
                $imageUrl = $defaultImage[$productKey]['url_zoom'];
            }
            foreach ($singleVar['option_values'] as $optionVal) {              
                $productImages = $this->getVariantSideImages($defaultImage, $optionVal['label'], $size_look, $imageUrl);
                $sides = [];
                foreach ($productImages as $key => $valueImage) {
                    $sides[$key]["image"] = array(
                        "src" => $productImages[$key]['src'],
                        "thumbnail" => $productImages[$key]['thumbnail'],
                    );
                }
                $allData = array(
                    'id' => $optionVal['id'],
                    'name' => $optionVal['label'],
                    'attribute_id' => $optionVal['option_display_name'],
                    'variant_id' => $singleVar['id'],
                    'inventory' => [
                        "stock" => $singleVar['inventory_level'],
                        "min_quantity" => 1,
                        "max_quantity" => 1000,
                        "quantity_increments" => 1
                    ],
                    'price' => $singleVar['calculated_price'],
                    'sides' => $sides,
                    "tier_prices" => $tierDetails
                );
                if($filter['color_type'] == "all" && strtolower($optionVal['option_display_name']) == 'color'){
                    $checkOption[] = $allData; 
                }
                if (strtolower($optionVal['option_display_name']) == 'color' && !in_array($optionVal['label'], $uniqueColors)) {
                    $uniqueColors[] = $optionVal['label'];
                    $thisCategory = $allData;
                    $checkOption[] = $thisCategory;
                }
                if (strtolower($optionVal['option_display_name']) == 'size' && !in_array($optionVal['label'], (array)$uniqueSizes)) {
                    $uniqueSizes[] = $optionVal['label'];
                    $thisCategory = $allData;
                    $checkOptionNew[] = $thisCategory;
                }
            }
        }
        if (empty($checkOption)) {
            return $checkOptionNew;
        } else {
            return $checkOption;
        }
    }
 
    /**
     * GET: all variant list
     * 
     * @author sonali@imprintnext.com
     * @date 05 feb 2022
     * @param $productId requested and variant id
     * @return array of tier price details
     */
    public function getvarientTierDiscounts($productId,$variant){
        $variantDetails = $this->call('GET', 'catalog/products/'.$productId.'/variants/'. $variant);
        $variantDetails = json_clean_decode($variantDetails, true);
        $variantData = $variantDetails['data'];
        if(!empty($variantData['price'])){
            $price = $variantData['price'];
        }else{
            $price = $variantData['calculated_price'];
        }
        $tierPrice = $this->call('GET', 'catalog/products/'.$productId.'/bulk-pricing-rules');
        $bulkPriceArr = json_decode($tierPrice,true);
        foreach($bulkPriceArr['data'] as $bulkValue){
            if($bulkValue['type'] == "fixed"){
                $percentage = $bulkValue['amount']/$price;
                $productQuantity = $bulkValue['quantity_min'];
                $productPrice = ($price * $percentage);
                $total = $price - $productPrice;
                $res = array(
                    "quantity" => $productQuantity,
                    "percentage" => $percentage,
                    "price" => $total
                );
                $tier[] = $res;
            }else{
                $percentage = $bulkValue['amount']; //10%
                $productQuantity = $bulkValue['quantity_min'];//5
                $productPrice = ($price * $percentage)/100;
                $total = $price - $productPrice;
                $res = array(
                    "quantity" => $productQuantity,
                    "percentage" => $percentage,
                    "price" => $total
                );
                $tier[] = $res;
            }
        }
         return $tier;
    }

    /**
     * Getting only color attribute list from store
     * 
     * @author divya@imprintnext.com
     * @date 28 Feb 2022
     * @param null
     * @return array of color attribute value
     */
    public function getColorAttributes() {
        $params['limit'] = 200;
        $products = $this->call('GET', 'catalog/products', $params);
        $productArray = json_clean_decode($products, true);
        //$i = 0;
        $attributeArray = array();
        $uniqueColors = array();
        foreach($productArray['data'] as $key => $prodId)
        {
            $proid[$key] = $prodId['id'];  
            $optionValue = $this->call('GET', 'catalog/products/'.$proid[$key].'/options', $params);
            $optionArray = json_clean_decode($optionValue, true);
            $attributeArray[] = $optionArray;
        }
        $colorsData = array(); 
        foreach($attributeArray as $attrValue){
           foreach($attrValue['data'] as $attrData){
                foreach($attrData['option_values'] as $optionVal){
                    if(strtolower($attrData['display_name']) == strtolower('color') && !in_array($optionVal['value_data']['colors'][0],$uniqueColors)){
                        $uniqueColors[] = $optionVal['value_data']['colors'][0];
                    $colors = array(
                         'id' => $optionVal['id'],
                         'attribute_id' => $optionVal['label'],
                         'name' => $optionVal['label'],
                         'hex_code' => $optionVal['value_data']['colors'][0],
                         'file_name' => "",
                         'color_type' => ""
                    );  
                   $colorsData['data'][] =  $colors;
                }
             }
           } 
        }
         return $colorsData;
    }
   
    
    /**
     * POST catlog product into store
     *
     * @author sonali@imprintnext.com
     * @date 11 Mar 2022
     * @param null
     * @return array catlogProduct created.
     */

    public function addCatalogProductToStore($priceInfo, $product, $price, $catalogPrice, $storeCategories)
    {
        if (!empty($product)) {
            try {
                $priceInfoArr = json_decode($priceInfo['product_data'], true);
                // denotes a product will be shown on the designer tool
                $showInDesignerId = $this->getShowInDesignerId();
                foreach ($priceInfoArr as $infoValue) {
                    // categories selected while import product
                    $selectedCategories = $infoValue['categories'];
                    if (empty($selectedCategories)) {
                        // create / add to default catalog category if no category selected
                        $categoryName = "catlog_products";
                        $createCatagory = $this->createCollection($categoryName);
                        $selectedCategories = [$createCatagory[2]];
                    }
                    $categories = array_merge([$showInDesignerId], $selectedCategories);
                }

                foreach ($product['variations'] as $variant) {
                    $varantImg =  !empty($variant['image_path']) ? $variant['image_path'][0] : '';
                    $thisVarPrice = $variant['piece_price'];
                    if (!empty($priceInfo['is_margin'])) {
                        if ($priceInfo['is_margin'] == 2) {
                            $thisVarPrice = $variant['piece_price'] + $priceInfo['product_margin'];
                        } else {
                            $percentagePrice = $variant['piece_price'] * ($priceInfo['product_margin'] / 100);
                            $thisVarPrice = $variant['piece_price'] + $percentagePrice;
                        }
                    } else {
                        $thisVarPrice = $price;
                    }
                    $optionValues = [
                        [
                            "label" => $variant['attributes']['size'],
                            "option_display_name" => 'size'
                        ],
                        [
                            "label" => $variant['attributes']['color'],
                            "option_display_name" => 'color'
                        ]
                    ];
                    if (isset($variant['attributes']['stitch_color'])) {
                        $optionValues[] = [
                            "label" => $variant['attributes']['stitch_color'],
                            "option_display_name" => 'stitch_color'
                        ];
                    }
                    $thisVar = [
                        "option_values" => $optionValues,
                        "sku" => (string)$variant['sku'],
                        "inventory_level" => $variant['quantity'],
                        "inventory_tracking " => 100,
                        "price" => $thisVarPrice,
                        "categories" => $categories,
                        "image_url" => $varantImg,
                        "gtin" => !empty($variant['gtin']) ? strval($variant['gtin']) : '',
                        "upc" => !empty($variant['gtin']) ? strval($variant['gtin']) : ''
                    ];
                    $variantArr[] = $thisVar;
                }
                if (!empty($product['images'])) {
                    $productImages = [
                        ["image_url" => $product['images']['src'], "is_thumbnail" => true, "sort_order" => 1]
                    ];
                } else {
                    $productImages = array_reduce($product['variations'], function ($accumulator, $variant) {
                        if (!empty($variant['image_path'])) {
                            $accumulator[] = [
                                "image_url" => $variant['image_path'][0], "is_thumbnail" => true, "sort_order" => count($accumulator) + 1
                            ];
                        }
                        return $accumulator;
                    });
                }
                $productArray = array(
                    "name" => $this->bigCommerce_body(addslashes($product['name'])),
                    "weight" => 100,
                    "description" => $this->bigCommerce_body(addslashes($product['description'])),
                    "is_visible" => true,
                    "type" => "physical",
                    "sku" =>  (string)$product['sku'],
                    "inventory_level" => $variant['quantity'],
                    "inventory_tracking " => 100,
                    "price" => $variant['piece_price'],
                    "categories" => $categories,
                    "images" => $productImages
                );
                if (!empty($variantArr)) {
                    $productArray['variants'] = $variantArr;
                }

                $productArray['module'] = 'product';
                $newProduct = $this->call('POST', 'catalog/products', $productArray);
                $newProduct = json_decode($newProduct, true);
                $newProduct =  $newProduct['data'];
            } catch (\Exception $e) {
                $newProduct = [
                    'status' => 0,
                    'message' => 'Invalid request',
                    'exception' => $e->getMessage()
                ];
            }
            return $newProduct['id'];
        } else {
            throw new \Exception("pass the required field");
        }
    }

    /**
     * Counting total no of products from store(Numeric value).
     * 
     * @author divya@imprintnext.com
     * @date 20 May 2022
     * @param filter parameters
     * @return integer (Total no of products numeric value)
     */
    public function getTotalProductsCount() {
        $categoriesId = $this->getShowInDesignerId();
        $params['categories:in'] = $categoriesId;
        $params['limit'] = 250;
        $totalProducts = $this->call('GET', 'catalog/products', $params);
        $totalProducts = json_clean_decode($totalProducts, true);
        return $totalProducts['meta']['pagination']['total'];
    }

    public function getBigcommerceProductInfo($productID){
        $productParams['include']  = 'images,custom_fields,variants';
        return  $this->call('GET', 'catalog/products/'.$productID, $productParams, "v3", "api");
    }
    public function getProductImageInfo($productID){
        return $this->call('GET', 'catalog/products/'.$productID .'/images', array(), "v3", "api");
    }
    
    public function getProductSingleVariantInfo($productID,$variantID){
        return $this->call('GET', 'catalog/products/'.$productID .'/variants/'.$variantID, array(), "v3", "api");
    }
    
    /* getting store related information */
    
    public function getStoreInformation()
    {
        $storeData = $this->call('GET', 'store', [], 'v2');
        $storeData = json_clean_decode($storeData, true);
        $storeUrl = $storeData['secure_url'];
        return $storeUrl;
    }
    
    /** ************************* Product Module End **************************** **/

    /** ************************* Customer Module Start **************************** **/

    /**
     * Get list of Customer
     *
     * @author     divya@imprintnext.com
     * @date       31st January 2022
     * @param  API filter params
     * @response   Array of all customer list
     */
    public function getAllCustomers($filters) {
        if (strpos($filters['searchString'], '@')){
            $params['email:in'] = $filters['searchString'];
        } 
        else{
            $params['name:like'] = $filters['searchString'];
        }
        $users = array();
        $params['limit'] = $filters['limit'];
        $params['page'] = isset($filters['page']) ? $filters['page'] : 1 ;
        if (!empty($filters['from_date']) && !empty($filters['to_date'])) {
            $params['date_created:min'] = date('Y-m-d', strtotime($filters['from_date']));
            $params['date_created:max'] = date('Y-m-d', strtotime($filters['to_date']));
        }
        if(!empty($filters['orderby']) && $filters['orderby'] == 'id'){
            $params['sort'] = 'date_created:'.$filters['order'];
        }   
        $customerData = $this->call('GET', 'customers',$params);
        $customerArray = json_decode($customerData, true);
        $totalCustomer = $customerArray['meta']['pagination']['total'];
        foreach ($customerArray['data'] as $cust) {
            $filter['customer_id'] = $cust['id'];
            $orderDetails = $this->call('GET', 'orders',$filter,'v2');
            $orderArray = json_clean_decode($orderDetails, true);
            $orderCount = count((array) $orderArray);
            $lastOrder = $orderArray[0]['id'];
            if ($filters['customer_no_order'] != 'false' && $filters['customer_no_order'] != '') {
                if ($orderCount < 1) {
                    $user = array();
                    $user['id'] = $cust['id'];
                    $user['first_name'] = $cust['first_name'];
                    $user['last_name'] = $cust['last_name'];
                    $user['email'] = $cust['email'];
                    $user['total_orders'] = $orderCount;
                    $user['last_order_id'] = str_replace('#', '', $lastOrder);
                    $user['date_created'] = date('Y-m-d h:i:s', strtotime($cust['date_created']));
                    $users['customer_list'][] = $user;
                } 
            }
            elseif(!empty($filters['customer_no_order']) && $filters['customer_no_order'] == 'false') {
                if ($orderCount > 0) {
                $user = array();
                $user['id'] = $cust['id'];
                $user['first_name'] = $cust['first_name'];
                $user['last_name'] = $cust['last_name'];
                $user['email'] = $cust['email'];
                $user['total_orders'] = $orderCount;
                $user['last_order_id'] = str_replace('#', '', $lastOrder);
                $user['date_created'] = date('Y-m-d h:i:s', strtotime($cust['date_created']));
                $users['customer_list'][] = $user;
                }    
        } else{
                $user['id'] = $cust['id'];
                $user['first_name'] = $cust['first_name'];
                $user['last_name'] = $cust['last_name'];
                $user['email'] = $cust['email'];
                $user['total_orders'] = $orderCount;
                $user['last_order_id'] = str_replace('#', '', $lastOrder);
                $user['date_created'] = date('Y-m-d h:i:s', strtotime($cust['date_created']));
                $users['customer_list'][] = $user;
            }
            $users['total_user'] = $totalCustomer;    
        }
      return $users;
    }
    /**
     * Get details of Customer
     *
     * @author     divya@imprintnext.com
     * @date       31st January 2022
     * @param  $customerID selected cutomer ID
     * @response   Array single customer details
     */
    public function getCustomerDetails($args) {
        $billingAddress = [];
        $orderDetails = [];
        $filter['id:in'] = $args;
        $customParam['customer_id:in']= $args;
        $customers = $this->call('GET', 'customers',$filter);
        $customerArray = json_decode($customers, true);
        $email = $customerArray['data'][0]['email'];
        $customerAddress = $this->call('GET', 'customers/addresses',$customParam);
        $addressArray = json_decode($customerAddress, true);
        foreach($addressArray['data'] as $customerData){
        $billingAddress = [
            'first_name' => $customerData['first_name'],
            'last_name' => $customerData['last_name'],
            'email' => $email,
            'phone' => $customerData['phone'],
            'address_1' => $customerData['address1'],
            'address_2' => $customerData['street_2'],
            'city' => $customerData['city'],
            'state' => $customerData['state_or_province'],
            'postcode' => $customerData['postal_code'],
            'country' => $customerData['country_code'],
        ];
        if (!empty($addressArray['data'])) {
            foreach ($addressArray['data'] as $key => $address) {
                $state = $this->getProvibce($address['country_code']);
                    foreach($state as $staValue){
                        if($staValue['state_name'] == $address['state_or_province']){
                            $stateCode = $staValue['state_code'];
                        }
                        
                    }
                    $shippingAddress[$key] = [
                        'id' => $address['id'],
                        'first_name' => $address['first_name'],
                        'last_name' => $address['last_name'],
                        'email' => $email,
                        'phone' => $customerData['phone'],
                        'address_1' => $address['address1'],
                        'address_2' => $address['address2'],
                        'city' => $address['city'],
                        'state_name' => $address['state_or_province'],
                        'state' => $stateCode,
                        'postcode' => $address['postal_code'],
                        'country_name' => $address['country'],
                        'country' => $address['country_code'],
                    ];
            }
         
        }else{
            $shippingAddress = array($billingAddress);
        }
    }

    $customerOrderDetails = $this->getOrderDetailsByCustomer($args);
    $customerOrder = json_decode($customerOrderDetails, true);
    $totalOrderOfCustomer = count($customerOrder);
    if (!empty($customerOrder)) {
        $totalAmount = 0;
        foreach ($customerOrder as $orderData) {
            $order = array();
            $order['id'] = $orderData['id'];
            $order['number'] = $orderData['id'];
            $order['quantity'] = array_sum(array_column($orderData['consignments'][0]['shipping'][0]['line_items'], 'quantity'));
            $order['currency'] = $orderData['currency_code'];
            $order['total_amount'] = (float)round($orderData['subtotal_inc_tax'],2);
            $order['created_date'] = date('Y-m-d h:i:s', strtotime($orderData['date_created']));
            $totalAmount += $order['total_amount'];
            $orderParams = [];
            $getItems = $this->call('GET', 'orders/'. $order['id'].'/products',$orderParams ,'v2');
            $getItemArr = json_decode($getItems,true);
            foreach($getItemArr as $itemValue){
            $productId = $itemValue['product_id'];
            $varientId = $itemValue['variant_id'];
            $productImage = $this->call('GET', 'catalog/products/'.$productId.'/images');
            $productImage = json_decode($productImage,true);
            if($productImage['status'] != 404) {
                $productImage = $productImage['data'];
                $productDefaultImage = array_column($productImage, 'is_thumbnail');
                $productKey = array_search( 1,$productDefaultImage);
                $variantDefaultImage = $productImage[$productKey]['url_zoom'];
                $variantSideImages = $this->getVariantSideImages($productImage,$color_lookup,$size_look,$variantDefaultImage);
            }

                $cartId = $orderData['cart_id'];
                if(!empty($cartId)){
                    $bigCart = new BigcommerceCart();
                    $sql = $bigCart->where("cart_id" ,'=',$cartId)
                    ->where("product_id" ,'=',$productId)
                    ->where("variant_id" ,'=',$varientId)
                   ->select('custom_design_id')->get()->toArray();
                   
                    if($sql == []){
                        $refID = 0;
                    }else{
                        $refID = $sql[0]['custom_design_id'];
                    }
                }else{
                    $bigCart = new BigcommerceCart();
                    $sql = $bigCart->where("order_id" ,'=',$order['id'])
                    ->where("product_id" ,'=',$productId)
                    ->where("variant_id" ,'=',$varientId)
                   ->select('custom_design_id')->get()->toArray();
                    if($sql == []){
                        $refID = 0;
                    }else{
                        $refID = $sql[0]['custom_design_id'];
                    }
                }

            $key =[];
            foreach ($itemValue['product_options'] as $optionValue) {
                if(strtolower($optionValue['display_name']) == 'size'){
                    $sizename = $optionValue['display_value_customer'];
                }
                if(strtolower($optionValue['display_name']) == 'color'){
                    $colorName = $optionValue['display_value_customer'];
                }
                $keyName = $optionValue['display_value_customer'];
                $key[]= $keyName;
            }
            $productVariant = implode(" , ",$key);
            $lineItems = [
                "id" => $itemValue['id'],
                "product_id" => $itemValue['product_id'],
                "variant_id" => $itemValue['variant_id'],
                "name" => (!empty($itemValue['product_options'])) ? $itemValue['name'].' - '.$productVariant : $itemValue['name'],
                "price" => (float)round($itemValue['base_price'],2),
                "quantity" => $itemValue['quantity'],
                "total" => (float)round(($itemValue['base_price'] * $itemValue['quantity']),2),
                "sku" => $itemValue['sku'],
                "custom_design_id" => $refID,
                "size" =>  $sizename,
                "color" => $colorName,
                'images' => $refID > 0 ? $this->getStoreProductImages($refID, $itemValue['variant_id']) : $variantSideImages,
                'productDecorationSettingData' => '',
                'weight_unit' => 'kg',
                'weight' => $itemValue['weight'],
            ];
            $order['lineItems'][] = $lineItems;
        }
            if (!empty($getItemArr)) {
                $orderDetails[] = $order;
            }
        }
    }
        return  [
            "id" => $customerData['customer_id'],
            "first_name" => $customerData['first_name'],
            "last_name" => $customerData['last_name'],
            "email" => $email,
            "profile_pic" => '',
            "phone" => $customerData['phone'],
            "total_orders" => $totalOrderOfCustomer,
            "total_order_amount" => (float)round($totalAmount,2),
            "average_order_amount" => strval(round($totalAmount / $totalOrderOfCustomer, 2)),
            "last_order" => $orderDetails[0]['created_date'],
            "date_created" => date('Y-m-d h:i:s', strtotime($customerData['created_at'])),
            "billing_address" => $billingAddress,
            "shipping_address" => $shippingAddress,
            "orders" => $orderDetails
        ];
    }

    /**
     * Get total count of availabel Customer
     *
     * @author     divya@imprintnext.com
     * @date       31st January 2022
     * @response   All customer count from store
     */
    public function storeCustomerCount(){
        $params['limit'] = 100;
        $params['page'] = 10;
        $customer = $this->call('GET', 'customers',$params);
        $customerData = json_decode($customer, true);       
        return $customerData['meta']['pagination']['total'];
    }

   /**
     * Get list of countries
     *
     * @author     divya@imprintnext.com
     * @date       02nd February 2022
     * @response   All country details
     */
    public function getShopCountries(){
        $countryParams['page'] = 1;
        $countryParams['limit'] = 233;
        $countries = $this->call('GET', 'countries',$countryParams,'v2');
        $countryArray = json_clean_decode($countries, true);
        $allCountries = array();
        foreach ($countryArray as $country) {
                $allCountries[] = array('countries_code' => $country['country_iso2'], 'countries_name' => $country['country']);
        }
        return $allCountries;
    }

   /**
     * Get list of all countrie's state
     *
     * @author     divya@imprintnext.com
     * @date       02nd February 2022
     * @response   State name and details
     */
    public function getProvibce($countryCode){
        $countryParams['page'] = 1;
        $countryParams['limit'] = 233;
        $countries = $this->call('GET', 'countries',$countryParams,'v2');
        $countryArray = json_clean_decode($countries, true);
        foreach($countryArray as $countryVal){
              if($countryVal['country_iso2'] == $countryCode || $countryVal['country'] == $countryCode){
                $code = $countryVal['id'];  
              }     
        }
        $states = array();
        $stateData = $this->call('GET', 'countries/'.$code.'/states',$countryParams,'v2');
        $stateDataArray = json_clean_decode($stateData, true);
        foreach ($stateDataArray as $stateInfo) {
           $stateVal = array('state_code' => $stateInfo['state'],
                              'state_name' => $stateInfo['state']);
                $states[] = $stateVal;
       }
       return $states;
    }

     /**
     * Delete customer
     *
     * @author     divya@imprintnext.com
     * @date       02nd February 2022
     * @response   json response
     */
    public function deleteShopCustomer($customerID){
         foreach ($customerID as $cusID){
            $customParam['id:in'] = $cusID;
            $this->call('DELETE', 'customers', $customParam); // delete customer
         }
        return ['status' => 1, 'message' => "customer deleted"];
    }

     /**
     * Create new customer
     *
     * @author     divya@imprintnext.com
     * @date       02nd February 2022
     * @response   return customer address
     */
    public function newShopCustomer($allPostPutVars){
        if(!empty($allPostPutVars)){
            $customerArr = array();
            $customerInfo['accepts_product_review_abandoned_cart_emails'] = true;
            $customerInfo['address_count'] = 1;
            $customerInfo['addresses'] = array(array(
                'address1' => $allPostPutVars['billing_address_1'],
                'address2' => $allPostPutVars['billing_address_2'],
                'city' => $allPostPutVars['billing_city'],
                'company' => $allPostPutVars['company_name'],
                'country_code' => $allPostPutVars['billing_country_code'],
                'customer_id' => $allPostPutVars['customer_id'],
                'first_name' => $allPostPutVars['first_name'],
                'last_name' => $allPostPutVars['last_name'],
                'phone' => $allPostPutVars['billing_phone'],
                'postal_code' => $allPostPutVars['billing_postcode'],
                'state_or_province' => $allPostPutVars['billing_state_code']
            ), array(
                'address1' => $allPostPutVars['shipping_address_1'],
                'address2' => $allPostPutVars['shipping_address_2'],
                'city' => $allPostPutVars['shipping_city'],
                'country_code' => $allPostPutVars['shipping_country_code'],
                'first_name' => $allPostPutVars['first_name'],
                'last_name' => $allPostPutVars['last_name'],
                'postal_code' => $allPostPutVars['shipping_postcode'],
                'state_or_province' => $allPostPutVars['shipping_state_code'],
            ));
            $customerInfo['authentication'] = array(
                "new_password" => $allPostPutVars['user_password']
            );
            $customerInfo['channel_ids'] = array(
                1
            );
            $customerInfo['company'] = $allPostPutVars['company_name'];
            $customerInfo['email'] = $allPostPutVars['user_email'];
            $customerInfo['first_name'] = $allPostPutVars['first_name'];
            $customerInfo['last_name'] = $allPostPutVars['last_name'];
            $customerInfo['phone'] = $allPostPutVars['billing_phone'];
            $customerArr[] = $customerInfo;
            return $this->call('POST', 'customers', $customerArr);
        }else{
            throw new \Exception("post all the customer details");
        }
        
        
    }

    /**
     * Add new shipping address
     * 
     * @author sonali@imprintnext.com
     * @date 20 Feb 2022
     * @param address data
     * @return customer shipping address details
     */
    public function changeDefaultAddress($newAddress){

        $args = intval($newAddress['user_id']) ; 
        $status = array();
        $filter['id:in'] = $args;
        $customers = $this->call('GET', 'customers',$filter);
        $customerArray = json_decode($customers, true);
        $customerArray = $customerArray ['data'];

        if (!empty($newAddress)) {
            try{
                $newAdd = [
                    "first_name" => $customerArray[0]['first_name'],
                    "last_name" => $customerArray[0]['last_name'],
                    "company" => $newAddress['company'],
                    "address1" => $newAddress['address_1'],
                    "address2" => $newAddress['address_2'],
                    "city" => $newAddress['city'],
                    "state_or_province" => $newAddress['state'],
                    "postal_code"=> $newAddress['post_code'],
                    "country_code" => $newAddress['country'],
                    "phone" => $newAddress['mobile_no'],
                    "address_type" => "RESIDENTIAL",
                    "customer_id" => intval($newAddress['user_id']),
                ];
                $status[] =  $newAdd;
                $shippingAdd = $this->call('POST', 'customers/addresses', $status);
                $shippingAdd = json_decode($shippingAdd, true);
                $shippingAdd = $shippingAdd['data'];
            }catch(\Exception $e){
                $shippingAdd = [
                    'status' => 0,
                    'message' => 'Invalid request',
                    'exception' => $e->getMessage(),
                ];
            }
            return $shippingAdd;
        }else{
            throw new \Exception("post customer shipping address");
        }
        
    }
  
    /**
     * GET: all customer short info
     * 
     * @author sonali@imprintnext.com
     * @date 19 Feb 2022
     * @param $customerId
     * @return arrau list of customer short data details
     */
    public function customerShortData($customerId,$shipId, $isAddress = false){ 
        $custom = [];
        $customer = $this->call('GET', 'customers/'. $customerId,$custom,'v2');
        $customer = json_decode($customer,true);
        
        $customerAddress = $this->call('GET', 'customers/' . $customerId . '/addresses' ,$custom,'v2');
        $customerAddress = json_decode($customerAddress,true);
        $customerDetails = [
            "id" => $customer['id'],
            "email" => $customer['email'],
            "name" => $customer['first_name']. ' '. $customer['last_name'],
            "phone" => ($customer['phone'] != '') ? $customer['phone'] : '',
        ];
        if(!empty($customerAddress)){
            foreach ($customerAddress  as $value) {
                $shipAdrr[] = [
                    'id' => $value['id'],
                    'first_name' => $customer['first_name'],
                    'last_name' => $customer['last_name'],
                    'email' => $customer['email'],
                    'phone' => $customer['phone'],
                    'address_1' => $value['street_1'],
                    'address_2' => $value['street_2'],
                    'city' => $value['city'],
                    'state_name' => $value['state'],
                    'state' => $value['state'],
                    'postcode' => $value['zip'],
                    'country_name' => $value['country'],
                    'country' => $value['country_iso2'],
                ];
            }
        }
       
        $customerDetails['shipping_address'] = $shipAdrr;
        $customerDetails['billing_address'] = array_shift($shipAdrr);
        return array('customer' => $customerDetails);
    }

    /**
     * GET: all customer short info
     * 
     * @author sonali@imprintnext.com
     * @date 19 Feb 2022
     * @param $customerId, $allPostPutVars form field values
     * @return arrau list of customer update data details
     */
    public function updateCustomerAddressInShop($allPostPutVars, $id){
        $country = $this->getShopCountries();
        $state = $this->getProvibce($allPostPutVars['country']);
        foreach($country as $conValue){
            if($conValue['countries_code'] == $allPostPutVars['country']){
                $countryName = $conValue['countries_name'];
            }
        }
        foreach($state as $staValue){
            if($staValue['state_code'] == $allPostPutVars['state']){
                $stateName = $staValue['state_name'];
            }
            
        }
        
        if (!empty($allPostPutVars)) {
            $address = array(
                "first_name"=> $allPostPutVars['first_name'],
                "last_name"=> $allPostPutVars['last_name'],
                "company"=> $allPostPutVars['company'],
                "street_1"=> $allPostPutVars['address_1'],
                "street_2"=> $allPostPutVars['address_2'],
                "city"=> $allPostPutVars['city'],
                "state" => $stateName,
                "zip" => $allPostPutVars['post_code'],
                "country"=> $countryName,
                "phone" => $allPostPutVars['mobile_no'],
                "address_type" => "residential",
                
            ); 
            
            $hideProd = $this->call('PUT', 'customers/'.$allPostPutVars['user_id'].'/addresses/'.$id, $address ,'v2');
        }
        if ($hideProd['method'] !== "PUT") {
           return array("status" => 1, "message" => "address updated");
        }else{
            return array();
        }
    }

    /**
     * get customer saved search as groups 
     *
     * @author     divya@imprintnext.com
     * @date       03 March 2022
     * @param  query filter Array
     * @response   Array of customer groups
     */
    public function getCustomerSavedSearches($filters) {
        $countGroup = $this->call('GET', 'customer_groups/count', [], 'v2');
        $countGroup = json_clean_decode($countGroup, true);
        if (!empty($filters['savedSearchID'])){
            $groupId = $filters['savedSearchID'];
            $group = $this->call('GET', 'customer_groups/'.$groupId, array(), 'v2');
            $groupArray = json_clean_decode($group, true);
            $customerGroups = array("id_group" => $groupArray['id'], 'name' => $groupArray['name']);
        }else{
            $params['limit'] = $filters['perpage']; //10
            $params['page'] = isset($filters['page']) ? $filters['page'] : 1 ;
            $group = $this->call('GET', 'customer_groups', $params, 'v2');
            $groupArray = json_clean_decode($group, true);
            $customerGroups = array();
            foreach ($groupArray as $groupData) {
                $thisGroup = array("id_group" => $groupData['id'], 'name' => $groupData['name']);
                $customerGroups[] = $thisGroup;
            }
        }
        return  array("data" => $customerGroups,"total_records" => $countGroup['count']);
        
    }

    /**
     * get customer details by customer group id 
     *
     * @author     divya@imprintnext.com
     * @date       03 March 2022
     * @param  query filter Array
     * @response   Array of customer groups
     */
    public function getStoreCustomerByGroupId($savedSearchID, $returnCount = false) {
        $getCustomers = $this->call('GET', 'customers', $savedSearchID, 'v2');
        $customerArray = json_clean_decode($getCustomers, true);
        $thisGroupCustomers = array();
        if (!empty($customerArray)) {
            foreach ($customerArray as $customer) {
                if($customer['customer_group_id'] == $savedSearchID){
                $thisGroupCustomers[] = array('id' => $customer['id'], 'name' => $customer['first_name']." ". $customer['last_name'], 'email' => $customer['email']);
            }
          }
        }
        if ($returnCount){
            return count($thisGroupCustomers);
        }else{
            return $thisGroupCustomers;
        }
    }

    /**
     * Update customer group 
     *
     * @author     divya@imprintnext.com
     * @date       03 Feb 2022
     * @param  query filter Array
     * @response   Array of customer groups
     */
    public function updateCustomerSavedSearch($updateData) {
        $customerGroupID = $updateData['id'];
        $groupName = $updateData['name'];
        $customerId = $updateData['customer_id'][0];
        $groupData = $this->call('GET', 'customer_groups/'.$customerGroupID, array(), 'v2');
        $groupDataArr = json_clean_decode($groupData, true);
        $categoryType = $groupDataArr['category_access']['type'];
        if(empty($groupDataArr['discount_rules']))
          {
              $newGroupData = array(
                  'name' => $groupName,
                  "discount_rules" => [
                      array(
                          "type" => $categoryType,
                          "method" => 'fixed',
                          "amount" => 0
                          )
                        ]
                    );
        }
      else{
        $newGroupData = array(
            'name' => $groupName,
            "discount_rules" => [
                array(
                "type" => $groupDataArr['discount_rules'][0]['type'],
                "method" => $groupDataArr['discount_rules'][0]['method'],
                "amount" => $groupDataArr['discount_rules'][0]['amount']
               )
             ]
          );
       }
        $updateGroup = $this->call('PUT', 'customer_groups/'.$customerGroupID, $newGroupData, 'v2');
        $customerUpdateArray = array(
            "customer_group_id" => $customerGroupID
        );
        //update existing customer assignments to groups
        $this->call('PUT', 'customers/'.$customerId, $customerUpdateArray, 'v2');
        if(!empty($updateGroup)){
            $returnData = array('status' => 1, 'message' => 'CustomerGroup is updated in Bigcommerce store');
        }else{
            $returnData = array();
        }
        return $returnData;   
    }

    /**
     * create customer groups
     *
     * @author     divya@imprintnext.com
     * @date       03 Feb 2022
     * @param  query filter Array
     * @response   Array of customer groups
     */
    public function createcustomerSavedSearch($name,$customers) {
        try{
            $group = $this->call('GET', 'customer_groups', array(), 'v2');
            $groupArray = json_clean_decode($group, true);
            $categoryType = $groupArray[0]['category_access']['type'];
            $groupPostData = array(
                            'name' => $name,
                            'discount_rules' => [
                             array(
                                "type" => $categoryType,
                                "method" => 'fixed',
                                "amount" => 0
                              )
                            ]
                       );
            $addGroup = $this->call('POST', 'customer_groups', $groupPostData, "v2");
            $addGroupArray = json_clean_decode($addGroup, true);
            $newCustGrpID = $addGroupArray['id'];
            $customerGroupIdArray = array("customer_group_id" => $newCustGrpID);
            foreach($customers as $customerId){
                $cid = $customerId; 
                $this->call('PUT', 'customers/'.$cid, $customerGroupIdArray,"v2");
            }
        }catch(\Exception $e){
            $newCustGrpID = [
                'status' => 0,
                'message' => 'Invalid request',
                'exception' => $e->getMessage(),
            ];
        }
        return $newCustGrpID;
    }

     /** 
     * delete customer groups
     *
     * @author     divya@imprintnext.com
     * @date       04 Feb 2022
     * @param  query filter Array
     * @response   Array of customer groups
     */
    public function deleteCustomerSavedSearch($customerGroupId) {
        $groupDeleted = false;
        if(!empty($customerGroupId))
        {
            $groupDeleted = true;
            $this->call('DELETE', 'customer_groups/'.$customerGroupId, array(), 'v2');
        }
        return $groupDeleted;
    }

    /**
     * geting customerGroupId  from store
     * 
     * @author Sonali@imprintnext.com
     * @date   16 june 2021
     * @param customer $customerId
     * @return customer groupId
     */
    public function savedSearchIDFromCustID($customerId){
        $customParam = [];
        $customerGroupId = $this->call('GET', 'customers',$customParam);
        $customerGroupArray = json_decode($customerGroupId, true);
        $customerGroupArray = $customerGroupArray['data'];
        foreach($customerGroupArray as $groupId){
            if($groupId['id'] == $customerId){
               $custGroupId = $groupId['customer_group_id'];
            }
            
        }
       return  $custGroupId;
    }

    /** 
     * get details of single job card
     *
     * @author     divya@imprintnext.com
     * @date       23 March 2022
     * @param  query filter Array
     * @response   Array of single job card details
     */
    public function getOrderLineItemData($orderID, $orderItemId, $is_customer){
        $orderParams = [];
        $ordersArr = $this->call('GET', 'orders/'.$orderID ,$orderParams,'v2');
        $ordersArr = json_decode($ordersArr,true);
        $orderShippingAddress = $this->call('GET', 'orders/'.$orderID.'/shipping_addresses',$orderParams,'v2');
        $orderShippingAddress = json_decode($orderShippingAddress, true);
        $productArray = $this->call('GET', 'orders/'.$orderID.'/products' ,$orderParams,'v2');
        $productArray = json_decode($productArray, true);
        $cartId = $ordersArr['cart_id'];
        $orderId = $ordersArr['id'];
        $bigCart = new BigcommerceCart();
        $productId = $productArray[0]['product_id'];
        $variantId = $productArray[0]['variant_id'];
        if(!empty($cartId)){
            $sql = $bigCart->where("cart_id" ,'=',$cartId)
            ->where("product_id" ,'=',$productId)
            ->where("variant_id" ,'=',$variantId)
            ->select('custom_design_id')->get()->toArray();
            if($sql == []){
                $refID = '';
            }else{
                $refID = $sql[0]['custom_design_id'];
            }
        }else{
             $sql = $bigCart->where("order_id" ,'=',$orderId)
            ->where("product_id" ,'=',$productId)
            ->where("variant_id" ,'=',$variantId)
            ->select('custom_design_id')->get()->toArray();
            if($sql == []){
                $refID = '';
            }else{
                $refID = $sql[0]['custom_design_id'];
            }
            
        }
        foreach($productArray as $itemValue){
            foreach($itemValue['product_options'] as $optionValue){
                if(strtolower($optionValue['display_name']) == "size"){
                    $size = [
                        "id" => $optionValue['id'],
                        "name"  => $optionValue['display_value_customer'],
                        "attribute_id" => $optionValue['option_id'],
                        "hex-code" => ""
                    ];
                }
                if(strtolower($optionValue['display_name']) == "color"){
                    $color = [
                        "id" => $optionValue['id'],
                        "name"  => $optionValue['display_value_customer'],
                        "attribute_id" => $optionValue['option_id'],
                        "hex-code" => ""
                    ];
                }
            }
        }
        $productParams['include']  = 'images,custom_fields,variants';
        $productCatlog = $this->call('GET', 'catalog/products/'.$productId,$productParams);
        $product = json_clean_decode($productCatlog, true);
        $product = $product['data'];
        if (isset($product['images']) && count($product['images']) > 0) {
            foreach ($product['images'] as $prodImgKey => $prodImg) {
                if($prodImg['is_thumbnail'] == 1 && $prodImgKey > 0)
                {
                    $productImages[$prodImgKey]['id'] = $productImages[0]['id'];
                    $productImages[$prodImgKey]['src'] = $productImages[0]['src'];
                    $productImages[$prodImgKey]['thumbnail'] = $productImages[0]['thumbnail'];            
                    
                    $productImages[0]['id'] = $prodImg['id'];
                    $productImages[0]['src'] = $prodImg['url_zoom'];
                    $productImages[0]['thumbnail'] = $prodImg['url_tiny'];
                }
                else {
                    $productImages[$prodImgKey]['id'] = $prodImg['id'];
                    $productImages[$prodImgKey]['src'] = $prodImg['url_zoom'];
                    $productImages[$prodImgKey]['thumbnail'] = $prodImg['url_tiny'];
                }
            }
        }
        $productDefaultImage = $productImages[0]['src'];
        $productImages = $this->getVariantSideImages($product['images'],$color['name'],$size['name'],$productDefaultImage);

        $orderData = array();
        $orderData['order_id'] = $orderID;
        $orderData['order_number'] = $orderID;
        $orderData['item_id'] = $orderItemId; 
        $orderData['product_id'] = $productArray[0]['product_id'];
        $orderData['variant_id'] = $productArray[0]['variant_id'];
        $orderData['name'] = $productArray[0]['name'];
        $orderData['quantity'] = $productArray[0]['quantity'];
        $orderData['sku'] = $productArray[0]['sku'];
        $orderData['total'] = $productArray[0]['total_inc_tax'];
        $image = $refID > 0 ? $this->getStoreProductImages($refID, $productArray[0]['variant_id']) : $productImages;
        $orderData['images'][0] = $image[0];
        $orderData['categories'] = [];
        $orderData['price'] = $productArray[0]['base_price'];
        $orderData['total'] = $ordersArr['total_inc_tax'];
        $orderData['attributes'] = (!empty($size) || !empty($color)) ? ["color" => $color, "size" => $size,] : [];
        $orderData['custom_design_id'] = isset($is_customer) ? $is_customer : 0;
        $orderData['customer_id'] = $ordersArr['customer_id'];
        $orderData['customer_email'] = $ordersArr['billing_address']['email'];
        $orderData['customer_first_name'] = $ordersArr['billing_address']['first_name'];
        $orderData['customer_last_name'] = $ordersArr['billing_address']['last_name'];
        $orderData['billing'] = array(
                    'first_name' => $ordersArr['billing_address']['first_name'],
                    'last_name' => $ordersArr['billing_address']['last_name'],
                    'company' => $ordersArr['billing_address']['company'],
                    'address_1' => $ordersArr['billing_address']['street_1'],
                    'address_2' => $ordersArr['billing_address']['street_2'],
                    'city' => $ordersArr['billing_address']['city'],
                    'state' => $ordersArr['billing_address']['state'],
                    'country' => $ordersArr['billing_address']['country'],
                    'postcode' => $ordersArr['billing_address']['zip']);
        $orderData['shipping'] = array(
                    'first_name' => $orderShippingAddress[0]['first_name'],
                    'last_name' => $orderShippingAddress[0]['last_name'],
                    'company' => $orderShippingAddress[0]['company'],
                    'address_1' => $orderShippingAddress[0]['street_1'],
                    'address_2' => $orderShippingAddress[0]['street_2'],
                    'city' => $orderShippingAddress[0]['city'],
                    'state' => $orderShippingAddress[0]['state'],
                    'country' => $orderShippingAddress[0]['country'],
                    'postcode' => $orderShippingAddress[0]['zip']);
        $orderData['country'] = $orderShippingAddress[0]['country'];
        $orderData['postcode'] = $orderShippingAddress[0]['zip'];
    
        return $orderData;
 }
    
    /*     * ************************* Customer Module  End **************************** */

    /*     * ***************************** Order Module Start ***************************** */
    /**
     * creat order quote request 
     *
     * @author     sonali@imprintnext.com
     * @date       17 feb 2022
     * @param  quotation data Array
     * @response   Array of new order id
     */
    public function createStoreOrder($quoteData){
        if (!empty($quoteData)) {
            try{
                $custom = $products = [];
                $productId = $quoteData['product_data'][0]['product_id'];
                $statusId = $quoteData['status_id'];
                $customerId = $quoteData['customer_id'];
                $customer = $this->call('GET', 'customers/' . $customerId, $custom, 'v2');
                $customer = json_decode($customer, true);
                $customerAdd = $this->call('GET', 'customers/' . $customerId . '/addresses', $custom, 'v2');
                $customerAddress = json_decode($customerAdd, true);
                $isSticker = 0;
                foreach ($quoteData['product_data'] as $item) {
                    if($item['is_sticker_product'] == 1) {
                        $isSticker = 1;
                    $product = $this->call('GET', 'catalog/products/'.$item['product_id']);
                    $product = json_decode($product,true);
                    $product = $product['data'];
                        $cartData[] =[
                            'product_id' => $item['product_id'],
                            'sheet_name' => $item['sheet_name'],
                            'variant_id' => $product['base_variant_id'],
                            'added_price' => $item['updated_price'],
                            'design_cost' => $item['design_cost'],
                            'quantity' => $item['quantity']
                        ];
                    }
                    else {
                        $productPrice = $item['updated_price'] + $item['design_cost'];
                        $discountAmount = $quoteData['discount_type'] == "percentage" ? ($quoteData['discount_amount'] / 100) * $quoteData['design_total'] : $quoteData['discount_amount'];
                        $discountPrice =  round($discountAmount / $item['overall_quantity'],2) ; 
                        $discountPP = $productPrice - $discountPrice;
                        $tax = round((($quoteData['tax_amount'] / 100) * $discountPP),2);
                        $products[] = [
                            "name" => $item['product_name'],
                            "quantity" => $item['quantity'],
                            "variant_id" => $item['variant_id'],
                            "product_id" => $item['product_id'],
                            "price_inc_tax" => $productPrice,
                            "price_ex_tax" => $productPrice - $tax,
                        ];
                    }
                }
                $totalQuantity = !empty($cartData) ? array_sum(array_column($cartData,'quantity')) : [];
                foreach ($cartData as $key => $singleCartData) {
                    $newProductCreation[$key] = $this->createStickerVariant($singleCartData);
                    $productPrice = $singleCartData['design_cost'] + $cartData[$key]['added_price'];
                    $discountAmount = $quoteData['discount_type'] == "percentage" ? ($quoteData['discount_amount'] / 100) * $quoteData['design_total'] : $quoteData['discount_amount'];
                    $discountPrice =  round($discountAmount / $totalQuantity, 2);
                    $discountPP = $productPrice - $discountPrice;
                    $tax = round((($quoteData['tax_amount'] / 100) * $discountPP), 2);
                    $stickerProducts[] = [
                        "name" => $item['product_name'],
                        "quantity" => $cartData[$key]['quantity'],
                        "variant_id" => $newProductCreation[$key]['variant_id'],
                        "product_id" => $item['product_id'],
                        "price_inc_tax" => $productPrice,
                        "price_ex_tax" => $productPrice - $tax
                    ];
                }

                foreach ($customerAddress as $addValue) {
                    $billingAddress = [
                        "first_name" => $addValue['first_name'],
                        "last_name" => $addValue['last_name'],
                        "street_1" => $addValue['street_1'],
                        "city" => $addValue['city'],
                        "state" => $addValue['state'],
                        "zip" => $addValue['zip'],
                        "country" => $addValue['country'],
                        "country_iso2" => $addValue['country_iso2'],
                        "email" => $customer['email'],
                    ];
                }
                $thisOrder = [
                    "status_id" => $statusId,
                    "customer_id" => $customerId,
                    'billing_address' => $billingAddress,
                    'products' => ($isSticker == 1) ? $stickerProducts : $products,
                    'customer_message' => $quoteData['note'],
                    'staff_notes' => $quoteData['note'],
                    'base_shipping_cost' => $quoteData['shipping_amount'],
                    'discount_amount' => ($quoteData['discount_type'] == "percentage") ? ($quoteData['discount_amount'] / 100) * $quoteData['design_total'] : $quoteData['discount_amount'],
                ];
                $addOrder = $this->call('POST', 'orders', $thisOrder, 'v2');
                if ($isSticker == 1) {
                    $this->deleteStickerVariant($productId);
                }
                $addOrder = json_decode($addOrder, true);
                $bigCart = new BigcommerceCart();
                foreach ($quoteData['product_data'] as $key => $productValue) {
                    if ($productValue['is_sticker_product'] == 1) {
                        $variantId = $stickerProducts[$key]['variant_id'];
                    } elseif ($productValue['variant_id'] != 0) {
                        $variantId = $productValue['variant_id'];
                    } else {
                        $variantId = $this->getVariantId($productValue['product_id']);
                    }
                    $order = [
                        'product_id' => $productValue['product_id'],
                        'variant_id' => $variantId,
                        'cart_id' => "",
                        'line_item_id' => "",
                        'custom_design_id' => $productValue['custom_design_id'],
                        "order_id" => $addOrder['id'],
                    ];
                    $bigCart->insert($order);
                }
            }catch (\Exception $e) {
                $addOrder = [
                    'status' => 0,
                    'message' => 'Invalid request',
                    'exception' => $e->getMessage(),
                ];
            }
            return array("id" => $addOrder['id'], "order_number" => $addOrder['id']);
        }
        
    }

    public function deleteStickerVariant($pId){
        $optionsData = $this->call('GET', 'catalog/products/'.$pId.'/options');
        $optionsData = json_decode($optionsData,true);
        $optionsData = $optionsData['data'];
        $sheetNames = array_column($optionsData, 'display_name');
        $index = array_search('sheets', $sheetNames);
        $this->call('DELETE', 'catalog/products/'.$pId.'/options/'.$optionsData[$index]['id']);
    }

    /**
     * retrieve orders from store
     *
     * @author sonali@imprintnext.com
     * @date       2 feb 2022
     * @param array $callParams
     * @param string lastOrderId
     * @param int store
     * @param int range
     * @param date fromDate
     * @param date toDate
     * @return json
     */
    public function allOrders($callParams){
        $shop = $this->call('GET', 'store', [],'v2');
        $shopData = json_decode($shop, true);
        if (!empty($shopData['timezone']['name'])) {
            date_default_timezone_set($shopData['timezone']['name']);
        }
        $orderDetails = $this->orders_get($callParams);
        $orderData = [];
        if($callParams['is_customize'] == 1 && $callParams['order_status'] != 'archive'){
            foreach ($orderDetails as $DetailValue){
                if($DetailValue['is_customization'] == 1 && $DetailValue['status'] != "Completed"){
                    $result = array(
                        "id" => $DetailValue['id'],
                        "order_number" => $DetailValue['id'],
                        "status_id" =>   $DetailValue['status'] == 'Refunded' ? 'Awaiting Payment' : $DetailValue['status'],
                        "status" => $DetailValue['status_id'] == 4 ?  7 : $DetailValue['status_id'],
                        "customer_id" => $DetailValue['customer_id'],
                        "customer_first_name" => $DetailValue['billing_address']['first_name'],
                        "customer_last_name" => $DetailValue['billing_address']['last_name'],
                        "created_date" => $DetailValue['date_created'],
                        "order_total_quantity" => $DetailValue['items_total'],
                        "total_amount" => $DetailValue['subtotal_inc_tax'],
                        "production" => "",
                        "currency" => $DetailValue['currency_code'],
                        "is_customize" => $DetailValue['is_customization'],
                        "is_deleted" => $DetailValue['is_deleted'],
                    );
                    $orderData[] = $result;
                }
            }
        } if($callParams['order_status'] != "archive" && $callParams['is_customize'] == 0){
            foreach ($orderDetails as $DetailValue){
                if($DetailValue['status'] != "Completed"){
                    $resultArr = array(
                        "id" => $DetailValue['id'],
                        "order_number" => $DetailValue['id'],
                        "status_id" =>  $DetailValue['status'] == 'Refunded' ? 'Awaiting Payment' : $DetailValue['status'],
                        "status" => $DetailValue['status_id'] == 4 ?  7 : $DetailValue['status_id'],
                        "customer_id" => $DetailValue['customer_id'],
                        "customer_first_name" => $DetailValue['billing_address']['first_name'],
                        "customer_last_name" => $DetailValue['billing_address']['last_name'],
                        "created_date" => date("Y-m-d h:i:s", strtotime($DetailValue['date_created'])),
                        "order_total_quantity" => $DetailValue['items_total'],
                        "total_amount" => $DetailValue['subtotal_inc_tax'],
                        "production" => "",
                        "currency" => $DetailValue['currency_code'],
                        "is_customize" => $DetailValue['is_customization'],
                        "is_deleted" => $DetailValue['is_deleted'],
                    );
                    $orderData[] = $resultArr;
                }
            }
        }
        if($callParams['order_status'] == "archive"){
            foreach ($orderDetails as $DetailValue){
                if($DetailValue['status'] == "Completed"){
                    $archive = array(
                        "id" => $DetailValue['id'],
                        "order_number" => $DetailValue['id'],
                        "status_id" =>   $DetailValue['status'] == 'Refunded' ? 'Awaiting Payment' : $DetailValue['status'],
                        "status" => $DetailValue['status_id'] == 4 ?  7 : $DetailValue['status_id'],
                        "customer_id" => $DetailValue['customer_id'],
                        "customer_first_name" => $DetailValue['billing_address']['first_name'],
                        "customer_last_name" => $DetailValue['billing_address']['last_name'],
                        "created_date" => $DetailValue['date_created'],
                        "order_total_quantity" => $DetailValue['items_total'],
                        "total_amount" => $DetailValue['subtotal_inc_tax'],
                        "production" => "",
                        "currency" => $DetailValue['currency_code'],
                        "is_customize" => $DetailValue['is_customization'],
                        "is_deleted" => 1,
                    );
                    $orderData[] = $archive;
                }
            }
        }
        return $orderData;
    }

    /**
     * retrieve order details from store
     * 
     * @author sonali@imprintnext.com
     * @date       5 feb 2022
     * @param array $callParams
     * @param string orderIncrementId
     * @param int store
     * @return json
     */
    public function orderDetails($args) {
        $order_id = $args['id'];
        $ordersData = [];
        $orders = $this->call('GET', 'orders/'.$order_id ,$ordersData,'v2');
        $ordersArr = json_decode($orders,true);
        $filter['customer_id'] = $ordersArr['customer_id'];
        $orderDetails = $this->call('GET', 'orders',$filter,'v2');
        $orderArray = json_clean_decode($orderDetails, true);
       
        $billingAddress = [
            'first_name' =>$ordersArr['billing_address']['first_name'],
            'last_name' => ($ordersArr['billing_address']['last_name'] != '') ? $ordersArr['billing_address']['last_name'] : '',
            'email' => $ordersArr['billing_address']['email'],
            'phone' => ($ordersArr['billing_address']['phone'] != '') ? $ordersArr['billing_address']['phone'] : '',
            'address_1' => ($ordersArr['billing_address']['street_1'] != '') ? $ordersArr['billing_address']['street_1'] : '',
            'address_2' => ($ordersArr['billing_address']['street_2'] != '') ? $ordersArr['billing_address']['street_2'] : '',
            'country' => $ordersArr['billing_address']['country'],
            'state' => $ordersArr['billing_address']['state'],
            'city' => $ordersArr['billing_address']['city'],
            'postcode' => $ordersArr['billing_address']['zip'],
            'company' => ($ordersArr['billing_address']['company'] != '') ? $ordersArr['billing_address']['company'] : '',
            'country_code' => $ordersArr['billing_address']['country_iso2'],
            "state_code" => "",
        ];
        $orderParams = [];
        $shipping = $this->call('GET', 'orders/'.$order_id. '/shipping_addresses' ,$orderParams ,'v2');
        $shippingArr = json_decode($shipping,true);
        foreach($shippingArr as $shippingValue){
            $shippingAddress = [
                'first_name' => $shippingValue['first_name'],
                'last_name' => ($shippingValue['last_name'] != '') ? $shippingValue['last_name'] : '',
                'email' => $shippingValue['email'],
                'phone' => ($shippingValue['phone'] != '') ? $shippingValue['phone'] : '',
                'address_1' => ($shippingValue['street_1'] != '') ? $shippingValue['street_1'] : '',
                'address_2' => ($shippingValue['street_2'] != '') ? $shippingValue['street_2'] : '',
                'country' => $shippingValue['country'],
                'state' => $shippingValue['state'],
                'city' => $shippingValue['city'],
                'postcode' => $shippingValue['zip'],
                'company' => ($shippingValue['company'] != '') ? $shippingValue['company'] : '',
                'country_code' => $shippingValue['country_iso2'],
                "state_code" => "",      
            ];      
        }
        $getItems = $this->call('GET', 'orders/'. $order_id.'/products',$orderParams ,'v2');
        $getItemArr = json_decode($getItems,true);
        $LineItem = [];
        if(isset($args['ui'])){
            foreach($getItemArr as $itemValue){
                $productId = $itemValue['product_id'];
                $varientId = $itemValue['variant_id'];
                $productImage = $this->call('GET', 'catalog/products/'.$productId.'/images');	
                $productImage = json_decode($productImage,true);	
                $productImage = $productImage['data'];	
                $productDefaultImage = array_column($productImage, 'is_thumbnail');	
                $productKey = array_search( 1,$productDefaultImage);	
                $variantDefaultImage = $productImage[$productKey]['url_zoom'];	
                $variantSideImages = $this->getVariantSideImages($productImage,$color_lookup,$size_look,$variantDefaultImage);
                if($ordersArr['id'] = $order_id){
                    $cartId = $ordersArr['cart_id'];
                    if(!empty($cartId)){
                        $bigCart = new BigcommerceCart();
                        $sql = $bigCart->where("cart_id" ,'=',$cartId)
                        ->where("product_id" ,'=',$productId)
                        ->where("variant_id" ,'=',$varientId)
                    ->select('custom_design_id')->get()->toArray();
                    
                        if($sql == []){
                            $refID = 0;
                        }else{
                            $refID = $sql[0]['custom_design_id'];       
                        }
                    }else{
                        $bigCart = new BigcommerceCart();
                        $sql = $bigCart->where("order_id" ,'=',$order_id)
                        ->where("product_id" ,'=',$productId)
                        ->where("variant_id" ,'=',$varientId)
                    ->select('custom_design_id')->get()->toArray();
                        if($sql == []){
                            $refID = 0;
                        }else{
                            $refID = $sql[0]['custom_design_id'];       
                        }
                    }     
                }
                $key =[];
                foreach ($itemValue['product_options'] as $optionValue) {
                    if($optionValue['display_name'] == 'Size'){
                        $sizename = $optionValue['display_value_customer'];
                    }
                    if($optionValue['display_name'] == 'Color'){
                        $colorName = $optionValue['display_value_customer'];
                    }
                    if(strtolower($optionValue['display_name']) == 'stitch_color'){
                        $stitchColorName = $optionValue['display_value_customer'];
                    }
                    $keyName = $optionValue['display_value_customer'];
                    $key[]= $keyName;  
                }
                $productVariant = implode(" , ",$key);
                $item = [
                    "id" => $itemValue['id'],
                    "product_id" => $itemValue['product_id'],
                    "variant_id" => $itemValue['variant_id'],
                    "name" => (!empty($itemValue['product_options'])) ? $itemValue['name'].' - '.$productVariant : $itemValue['name'],
                    "price" => (float)round($itemValue['base_price'],2),
                    "quantity" => $itemValue['quantity'],
                    "total" => (float)round(($itemValue['base_price'] * $itemValue['quantity']),2),
                    "sku" => $itemValue['sku'],
                    "custom_design_id" => $refID,
                    "size" =>  $sizename,
                    "color" => $colorName,
                    "stitch_color" => $stitchColorName,
                    'images' => $refID > 0 ? $this->getStoreProductImages($refID, $itemValue['variant_id']) : $variantSideImages,
                    'productDecorationSettingData' => '',
                    'weight_unit' => 'kg',
                    'weight' => $itemValue['weight'],
                ];
                $LineItem[$refID]['line_items'][] = $item;
            }
            $LineItem = array_values($LineItem);
        }else {
            foreach($getItemArr as $itemValue){
                $productId = $itemValue['product_id'];
                $varientId = $itemValue['variant_id'];
                $productImage = $this->call('GET', 'catalog/products/'.$productId.'/images');	
                $productImage = json_decode($productImage,true);	
                $productImage = $productImage['data'];	
                $productDefaultImage = array_column($productImage, 'is_thumbnail');	
                $productKey = array_search( 1,$productDefaultImage);	
                $variantDefaultImage = $productImage[$productKey]['url_zoom'];	
                $variantSideImages = $this->getVariantSideImages($productImage,$color_lookup,$size_look,$variantDefaultImage);
                if($ordersArr['id'] = $order_id){
                    $cartId = $ordersArr['cart_id'];
                    if(!empty($cartId)){
                        $bigCart = new BigcommerceCart();
                        $sql = $bigCart->where("cart_id" ,'=',$cartId)
                        ->where("product_id" ,'=',$productId)
                        ->where("variant_id" ,'=',$varientId)
                    ->select('custom_design_id')->get()->toArray();
                    
                        if($sql == []){
                            $refID = 0;
                        }else{
                            $refID = $sql[0]['custom_design_id'];       
                        }
                    }else{
                        $bigCart = new BigcommerceCart();
                        $sql = $bigCart->where("order_id" ,'=',$order_id)
                        ->where("product_id" ,'=',$productId)
                        ->where("variant_id" ,'=',$varientId)
                    ->select('custom_design_id')->get()->toArray();
                        if($sql == []){
                            $refID = 0;
                        }else{
                            $refID = $sql[0]['custom_design_id'];       
                        }
                    }     
                }
                $key =[];
                foreach ($itemValue['product_options'] as $optionValue) {
                    if($optionValue['display_name'] == 'Size'){
                        $sizename = $optionValue['display_value_customer'];
                    }
                    if($optionValue['display_name'] == 'Color'){
                        $colorName = $optionValue['display_value_customer'];
                    }
                    if(strtolower($optionValue['display_name']) == 'stitch_color'){
                        $stitchColorName = $optionValue['display_value_customer'];
                    }
                    $keyName = $optionValue['display_value_customer'];
                    $key[]= $keyName;  
                }
                $productVariant = implode(" , ",$key);
                $item = [
                    "id" => $itemValue['id'],
                    "product_id" => $itemValue['product_id'],
                    "variant_id" => $itemValue['variant_id'],
                    "name" => (!empty($itemValue['product_options'])) ? $itemValue['name'].' - '.$productVariant : $itemValue['name'],
                    "price" => (float)round($itemValue['base_price'],2),
                    "quantity" => $itemValue['quantity'],
                    "total" => (float)round(($itemValue['base_price'] * $itemValue['quantity']),2),
                    "sku" => $itemValue['sku'],
                    "custom_design_id" => $refID,
                    "size" =>  $sizename,
                    "color" => $colorName,
                    "stitch_color" => $stitchColorName,
                    'images' => $refID > 0 ? $this->getStoreProductImages($refID, $itemValue['variant_id']) : $variantSideImages,
                    'productDecorationSettingData' => '',
                    'weight_unit' => 'kg',
                    'weight' => $itemValue['weight'],
                ];
                $LineItem[] =$item;   
            }
        }
            

         return [
            "id" => $ordersArr['id'],
            "order_number" =>$itemValue['order_id'],
            "customer_first_name" => $shippingValue['first_name'],
            "customer_last_name" => $shippingValue['last_name'],
            'customer_email' => $shippingValue['email'],
            "customer_id" => $ordersArr['customer_id'],
            "created_date" => date('Y-m-d H:i:s', strtotime($ordersArr['date_created'])),
            "total_amount" => (float)round($ordersArr['subtotal_inc_tax'],2),
            "total_tax" => (float)round($ordersArr['subtotal_tax'],2),
            "total_shipping" => !empty($ordersArr['base_shipping_cost']) ? (float)round($ordersArr['base_shipping_cost'],2) : (float)round($ordersArr['shipping_cost_tax'],2),
            "total_discounts" => (float)round($ordersArr['discount_amount'],2),
            "currency" => $ordersArr['currency_code'],
            "note" => ($ordersArr['staff_notes'] != '') ? $ordersArr['staff_notes'] : '',
            "production" => '',
            "status_id" => $ordersArr['status'] == 'Refunded' ? 'Awaiting Payment' : $ordersArr['status'],
            "status" => $ordersArr['status_id'] == 4 ?  7 : $ordersArr['status_id'],
            "total_orders" => count((array) $orderArray),
            "billing" => $billingAddress,
            "shipping" => $shippingAddress,
            "payment" => $ordersArr['payment_method'] ,
            "store_url" => "https://{$this->shopDomain}/",
            "orders" => $LineItem,
         ];
    }

    /**
     * GET: bigcommerce order details
     *
     * @author sonali@imprintnext.com
     * @date   10 Feb 2022   
     * @param $orderID requested order ID
     * @return Array order details
     */
    public function orderInfo($orderID){
        $orderParams =[];
        $ordersArr = $this->call('GET', 'orders/'.$orderID ,$orderParams,'v2');
        $ordersArr = json_decode($ordersArr,true);
        $orderProducts = $this->call('GET', 'orders/'.$orderID.'/products',$orderParams,'v2');
        $getItemArr = json_decode($orderProducts,true);
        $cartId = $ordersArr['cart_id'];
        $orderId = $ordersArr['id'];
        $bigCart = new BigcommerceCart();
        $lineItems = [];
        $orderDetails = [];
        foreach($getItemArr as $itemValue){
            $productId = $itemValue['product_id'];
            $variantId = $itemValue['variant_id'];
            if(!empty($cartId)){
                $sql = $bigCart->where("cart_id" ,'=',$cartId)
                ->where("product_id" ,'=',$productId)
                ->where("variant_id" ,'=',$variantId)
                ->select('custom_design_id')->get()->toArray();
                if($sql == []){
                    $refID = '';
                }else{
                    $refID = $sql[0]['custom_design_id'];
                }
            }else{
                
                $sql = $bigCart->where("order_id" ,'=',$orderId)
                ->where("product_id" ,'=',$productId)
                ->where("variant_id" ,'=',$variantId)
                ->select('custom_design_id')->get()->toArray();
                if($sql == []){
                    $refID = '';
                }else{
                    $refID = $sql[0]['custom_design_id'];
                }
                
            }
            if($refID > 0){	
                $artwork = "design_tool";	
            }else{	
                $artwork = "uploaded_file";	
            }
            foreach($itemValue['product_options'] as $optionValue){
                if(strtolower($optionValue['display_name']) == "size"){
                    $size = [
                        "id" => $optionValue['id'],
					    "name"  => $optionValue['display_value_customer'],
					    "attribute_id" => $optionValue['option_id'],
					    "hex-code" => ""
                    ];
                }
                if(strtolower($optionValue['display_name']) == "color"){
                    $color = [
                        "id" => $optionValue['id'],
					    "name"  => $optionValue['display_value_customer'],
					    "attribute_id" => $optionValue['option_id'],
					    "hex-code" => ""
                    ];
                }
                if(strtolower($optionValue['display_name']) == "stitch_color"){
                    $stitch = [
                        "id" => $optionValue['id'],
					    "name"  => $optionValue['display_value_customer'],
					    "attribute_id" => $optionValue['option_id'],
					    "hex-code" => ""
                    ];
                }
            }

            $productParams['include']  = 'images,custom_fields,variants';
            $productCatlog = $this->call('GET', 'catalog/products/'.$productId,$productParams);
            $product = json_clean_decode($productCatlog, true);
            $product = $product['data'];
            if (isset($product['images']) && count($product['images']) > 0) {
                foreach ($product['images'] as $prodImgKey => $prodImg) {
                    if($prodImg['is_thumbnail'] == 1 && $prodImgKey > 0)
                    {
                        $productImages[$prodImgKey]['id'] = $productImages[0]['id'];
                        $productImages[$prodImgKey]['src'] = $productImages[0]['src'];
                        $productImages[$prodImgKey]['thumbnail'] = $productImages[0]['thumbnail'];             
    
                        $productImages[0]['id'] = $prodImg['id'];
                        $productImages[0]['src'] = $prodImg['url_zoom'];
                        $productImages[0]['thumbnail'] = $prodImg['url_tiny'];
                    }
                    else {
                        $productImages[$prodImgKey]['id'] = $prodImg['id'];
                        $productImages[$prodImgKey]['src'] = $prodImg['url_zoom'];
                        $productImages[$prodImgKey]['thumbnail'] = $prodImg['url_tiny'];
                    }
                    
                }
            }
            $productDefaultImage = $productImages[0]['src'];
            $productImages = $this->getVariantSideImages($product['images'],$color['name'],$size['name'],$productDefaultImage);


            $items =[
                "item_id" => $itemValue['id'],
			    "product_id" => $itemValue['product_id'],
			    "variant_id" => $itemValue['variant_id'],
			    "product_name" => $itemValue['name'],
			    "product_sku" => $itemValue['sku'],
			    "quantity" => $itemValue['quantity'],
			    "price" => $itemValue['base_price'],
			    "images" => $refID > 0 ? $this->getStoreProductImages($refID, $itemValue['variant_id']): $productImages,
                "attributes" => [
                    "color" => $color,
                    "size" => $size,
                    "stitch_color" => $stitch,
                ],
                "color" => $color['name'],
			    "size" => $size['name'],
                "stitch_color" => $stitch['name'],
			    "ref_id" => $refID,
			    "artwork_type" => $artwork,
			   "design_cost" => 0,
			    "vendor_list" => [],
            ];
            $lineItems[] = $items;
        }
        $orderDetails['order_details'] = [
            "order_id" => $ordersArr['id'],
            "customer_id" => $ordersArr['customer_id'],
            "order_incremental_id" => $ordersArr['id'],
            "store_id" => 1,
            "order_items" => $lineItems,
            
        ];
        
         return json_encode($orderDetails);
        }

    /**
     * get orders from store
     *
     * @author sonali@imprintnext.com
     * @date       24 jan 2022
     * @param params order filter parameters
     * @param filter 
     */
     private function orders_get($params) {
        $filter['sort'] = $params['orderby'] . ':' . $params['order'];
        $fromDate = ($params['fromDate'] != '') ? date("Y-m-d\TH:i:s", strtotime($params['fromDate'])) : '';
        $toDate = ($params['toDate'] != '') ? date("Y-m-d\TH:i:s", strtotime($params['toDate'] . ' +1 day')) : '';
        $filter['limit'] = isset($params['per_page']) ? $params['per_page'] : 250;
        $filter['page'] =  $params['page'] ? $params['page'] : 1 ;
        if (isset($params['search']) && $params['search'] != '') {
            $filter['min_id'] = $params['search'];
            $filter['max_id'] = $params['search'];
        }
        if (isset($params['fromDate']) && isset($params['toDate']) && $params['fromDate'] != '' && $params['toDate'] != '') {
            $filter['min_date_created'] = $fromDate;
            $filter['max_date_created'] = $toDate;
        }
        $finalOrder = [];
        $orders = $this->call('GET', 'orders', $filter,'v2');
        $ordersArr = json_decode($orders,true);
        foreach($ordersArr as $orderValue){
            $orderId = $orderValue['id'];
            $cartId = $orderValue['cart_id'];
            $bigCart = new BigcommerceCart();
            if(!empty($cartId)){
                $sql = $bigCart->where("cart_id" ,'=',$cartId)->where("custom_design_id", ">",0)->select('custom_design_id','cart_id')->get()->toArray();
                if($sql == []){
                    $orderValue['is_customization'] = 0;
                }else{
                    $orderValue['is_customization'] = 1;
                }
                $finalOrder[] = $orderValue;

            }else{
                $sql = $bigCart->where("order_id" ,'=',$orderId)->select('custom_design_id','cart_id')->get()->toArray();
                if($sql == []){
                    $orderValue['is_customization'] = 0;
                }else{
                    $orderValue['is_customization'] = 1;
                }
                $finalOrder[] = $orderValue;
            }
        }
        return $finalOrder;
    }

    /**
     * GET: Bigcommerce store images
     *
     * @param $productId selected product ID
     * @return Array product details
     */
    public function getStoreProductImages($refID, $variantID) {
        $stateDesignPath = path('abs', 'design_state') . 'carts/' . $refID . '.json';
        $productImages = [];
        if (!file_exists($stateDesignPath)) {
            $stateDesignPath = path('abs', 'design_state') . 'predecorators/' . $refID . '.json';
        }
        if (!file_exists($stateDesignPath)) {
            $stateDesignPath = path('abs', 'design_state') . 'quotes/' . $refID . '.json';
        }
        if (!file_exists($stateDesignPath)) {
            $stateDesignPath = path('abs', 'design_state') . 'artworks/' . $refID . '.json';
        }
        if (file_exists($stateDesignPath)) {
            $jsonData = json_clean_decode(file_get_contents($stateDesignPath), true);
            if (!empty($jsonData)) {
                foreach ($jsonData['design_product_data'] as $designProductData) {
                    if ((is_array($designProductData)))
                     {
                        $customImageUrl = $designProductData['design_urls'];
                        break;
                    }
                }
                if (empty($customImageUrl)) {
                    $customImageUrl = $jsonData['design_product_data'][0]['design_urls'];
                }
                foreach ($customImageUrl as $key => $customImage) {
                    $img = [
                        "id" => ($key + 1),
                        "src" => $customImage,
                        "thumbnail" => $customImage,
                    ];
                    $productImages[] = $img;
                }
            }
            
        }
        return $productImages;
    }
  
    /**
     * retrieve order logs
     *
     * @author 
     * @date      
     * @param string orderID
     * @return array of order log from Shopify
     */
    public function getOrderLog($orderId){
        $orders = [];
        $order =  $this->call('GET', 'orders/'.$orderId,$orders,'v2');
        $orderarr = json_decode($order,true);
        $orderLogs = [];
        $storeLogs = [];
        $storeLogs[] = array(
            "order_id" => $orderarr['id'],
            "log_type" => "order_status",
            "message" => $orderarr['status'],
            "status" => "new",
            "created_at" => date("Y-m-d h:i:s", strtotime($orderarr['date_created'])),
            "updated_at" => date("Y-m-d h:i:s", strtotime($orderarr['date_modified'])) 
        );
        
        $storeLogs[] = array(
            "order_id" => 20515,
		    "message" => "Paid",
		    "date_paid" => date("Y-m-d h:i:s", strtotime($orderarr['date_created'])),
		    "payment_method" => $orderarr['payment_method'],
		    "payment_method_title" => "",
		    "log_type" => "Payment status",
		    "status" => "new",
		    "created_at" => date("Y-m-d h:i:s", strtotime($orderarr['date_created'])),
		    "updated_at" => date("Y-m-d h:i:s", strtotime($orderarr['date_modified']))
        );
        
        foreach ($storeLogs as $log) {
            if (is_array($log) && !empty($log)) {
                $log['agent_type'] = "admin";
                $log['agent_id'] = "";
                $log['status'] = "new";
               $orderLogs[] = $log;
            }
        }

        return json_encode($orderLogs);
    }

     /**
     * update order status
     *
     * @author sonali@imprintnext.com
     * @date       11 feb 2022
     * @param string orderID
     * @return boolean status
     */
    public function updateOrderStatuses($orderId, $orderData){
        $orderStatus = $orderData['statusKey'];
        $response = ['status' => 0 ];
        $param = [
            "status_id" => $orderStatus
        ];
        $orderResponse = $this->call("PUT", 'orders/' . $orderId,$param, "v2");
        if(!empty($orderResponse[0])){
            $response = ['status' => 0 ];
        }
        else {
            $response = ['status' => 1 ];
        }
        return $response;
    }

    /**
     * GET: Order line item details from store with attributes
     *
     * @author     divya@imprintnext.com
     * @date       03 January 2024
     * @param  orderId
     * @response   order line item details
     */
    public function getAttributeFromStore($orderId, $storeId) {
        $orderParams = [];
        $getItems = $this->call('GET', 'orders/'. $orderId.'/products',$orderParams ,'v2');
        $getItemArr = json_decode($getItems,true);
        $orderLineItemDetails = [];
        $orderKey = 0;
        foreach($getItemArr as $linItemInfo) {
            $orderLineItemDetails[$orderKey]['order_item_id'] = $linItemInfo['id'];
            $orderLineItemDetails[$orderKey]['product_id'] = $linItemInfo['product_id'];
            $orderLineItemDetails[$orderKey]['variant_id'] = $linItemInfo['variant_id'];
            foreach($linItemInfo['product_options'] as $optionsData) {
                $attributeName = $optionsData['display_name'];
                $attributeNameValName = $optionsData['display_value'];
                $attribute[strtolower($attributeName)] = $attributeNameValName;
            }
            $orderLineItemDetails[$orderKey]['attribute_obj'] = $attribute;
            $orderKey++;
        }
        return  $orderLineItemDetails;
    }

     /*     * ***************************** Order Module End ***************************** */
    /*     * ***************************** cart Module Start ***************************** */

    /**
     * GET custom preview images of cart items
     * 
     * @author inkXE
     * @date 20 Dec 2018
     * @param $refids design reference ID
     * @return Cart link
     */
    public function getDesignPreviewImages($variantID, $pid, $refids = 0)
    {
        try {
            if ($refids) {
                $regidArr = explode(',', $refids);
                $jsonData = '';
                $desStateDIR = path('abs', 'design_state') . 'carts';
                $baseImagePath = $desStateDIR.SEPARATOR;
                foreach ($regidArr as $values) {
                    $stateDesignFile = $baseImagePath . $values . '.json';
                    $jsonData = json_decode(file_get_contents($stateDesignFile), true);
                    if ($jsonData != '') {
                        foreach ($jsonData['design_product_data'] as $capture) {
                            if ($capture['variant_id'][0] == $variantID || $pid == $variantID) {
                                // $designStatus = 1;
                                $images = $capture['design_urls'];
                                break;
                            }
                        }
                    } else {
                        $msg = array("status" => "nodata");
                        return json_encode($msg);
                    }
                }
                return $images;
            }
        } catch (\Exception $e) {
            $result = array('Caught exception:' => $e->getMessage());
            return json_encode($result);
        }
    }

    /**
     * new addtocart api for fast response.
     * 
     * @author steve@imprintnext.com
     * @date 25 Jan 2022
     * @param product details
     * @return Cart link
     */
     public function createAddToCartLink($cartProduct, $designID, $cartItemId, $action, $cartId = '')
    {
        if(!empty($cartId)) {
            $cartItemList = $this->call('GET', 'carts/'.$cartId);
            $cartItemDetails = json_decode($cartItemList, true);
            $cartLineItemData = $cartItemDetails['data']['line_items']['physical_items'];
            
            foreach($cartLineItemData as $lineItemData) {
                if($lineItemData['product_id'] == $cartProduct[0]['product_id']) {
                  $lineItemId = $lineItemData['id'];
                }
            }
            $bigAddToCart = new BigcommerceCart();
            if(!empty($lineItemId)) {
                $cartData = $bigAddToCart->where("line_item_id", '=', $lineItemId)->delete();
                $this->call('DELETE', 'carts/' .$cartId.'/items/'.$lineItemId);
            }
            $isCartIdExist = $bigAddToCart->where("cart_id", '=', $cartId)
            ->select('line_item_id')->get()->toArray();
            if(empty($isCartIdExist)) {
                $cartId = 0;
            }
        }

        if(!empty($cartProduct)){
            try{
                $pid = $cartProduct[0]['product_id'];
                $tierPriceData = array();
                $productParams['include']  = 'images,custom_fields,variants';
                // Get tier price details
                $metaData = $this->call('GET', '/admin/products/' . $pid . '/metafields.json');
                $tierPriceData = array();
                $commonTierPrice = array();
                $variantTierPrice = array();
                if (!empty($metaData)) {
                    foreach ($metaData as $meta) {
                        if ($meta['namespace'] == 'imprint_data' && $meta['key'] == 'tier_content') { // tier = true
                            $tierContent = $meta['value'];
                            $tierPriceData = json_decode($tierContent, true);
                            break;
                        }
                    }
                    // little confusion with the key here but was done as sent from admin form field
                    if ($tierPriceData['pricing_per_variants'] == 'true') { //same for all Variants
                        foreach ($tierPriceData['price_rules'][0]['discounts'] as $discount) {
                            $commonTierPrice[] = array(
                                "upper_limit" => $discount['upper_limit'],
                                "lower_limit" => $discount['lower_limit'],
                                "discount" => $discount['discount'],
                                "discountType" => $tierPriceData['discount_type']
                            );
                        }
                    } else {
                        foreach ($tierPriceData['price_rules'] as $variantRule) {
                            foreach ($variantRule['discounts'] as $discount) {
                                $variantTierPrice[$variantRule['id']][] = array(
                                    "upper_limit" => $discount['upper_limit'],
                                    "lower_limit" => $discount['lower_limit'],
                                    "discount" => $discount['discount'],
                                    "discountType" => $tierPriceData['discount_type']
                                );
                            }
                        }
                    }
                }
                // get option array for new product
                $optionArr = array();
                $parentAttributes = $cartProduct[0]['options'];
                if (!empty($parentAttributes)) {
                    $position = 0;
                    foreach ($parentAttributes as $key => $opt) {
                        if (strpos($key, "_id") === false) {
                            $position++;
                            array_push($optionArr, array("name" => $key, "position" => $position));
                        }
                    }
                }
                // prepare variant array
    
                foreach ($cartProduct as $key => $cartVariant) {
                    if($cartVariant['is_sticker_product'] == 1){
                        $cartParams[$key] = $this->createStickerVariant($cartVariant);
                        $cartParams['include'] = 'redirect_urls';
                        $cartParams['module'] = 'cart';
                    } else {
                        $variantID = $cartVariant['variant_id'];
                        $quantity = $cartVariant['qty'];
                        if ($cartVariant['qty'] > 0) {
                            $quantity = $cartVariant['qty'];
                        }
                        if ($variantID == $pid) {
                            $productArray = $this->call('GET', 'catalog/products/' . $pid, $productParams);
                            $productArray = json_clean_decode($productArray, true);
                            $variantID = !empty($productArray['data']['base_variant_id']) ? $productArray['data']['base_variant_id'] : $productArray['data']['variants'][0]['id'];
                        }
                        $variant = $this->call('GET', 'catalog/products/' . $pid . '/variants/' . $variantID);
                        $variant = json_clean_decode($variant, true);
                        $variant = $variant['data'];
                        $variantPrice = (!empty($variant['price']) ? $variant['price'] : $variant['calculated_price']);

                        // Product Custom Price
                        if (isset($cartVariant['added_price']) && $cartVariant['added_price'] > 0) {
                            if (isset($cartVariant['is_variable_decoration']) && $cartVariant['is_variable_decoration'] > 0) {
                                $totalAddredPrice = $cartVariant['added_price'];
                            } else {
                                $totalAddredPrice = $variantPrice + $cartVariant['added_price'];
                            }
                            $cartParams[$key]['list_price'] = $totalAddredPrice;
                        }


                        $cartParams[$key]['quantity'] = $quantity;
                        $cartParams[$key]['product_id'] = $pid;
                        $cartParams[$key]['variant_id'] = $variantID;
                        $cartParams['include'] = 'redirect_urls';
                        $cartParams['module'] = 'cart';
                    }
                }
                $path = 'carts';
                $existingCart = [];
                if (isset($cartId) && $cartId != '' && $cartId != '0') {
                    $path .= "/" . $cartId . "/items";
                    if ($action == 'update' && isset($cartItemId) && $cartItemId != '' && $cartItemId != 0) {
                        $deletePath = $path . "/" . $cartItemId;
                        $cart = $this->call('DELETE', $deletePath, [], 'v3');
                    }
    
                    $bigAddToCart = new BigcommerceCart();
                    $cartData = $bigAddToCart->where("cart_id", '=', $cartId)
                    ->select('line_item_id')->get()->toArray();
                    foreach ($cartData as $key => $value) {
                        $existingCart[] = $value['line_item_id'];
                    }
                }
                $cart = $this->call('POST', $path, $cartParams, 'v3');
                $cart = json_clean_decode($cart, true);
                $cart = $cart['data'];
                $line_items = $cart['line_items'];
                $physical_items = $line_items['physical_items'];
                if ($designID != 0) {
                    foreach ($physical_items as $key => $value) {
                        $cartItemId =  $value['id'];
                        $productId =  $value['product_id'];
                        $variant_Id =  $value['variant_id'];
                        foreach ($cartProduct as $pIds) {
                            if ($pIds['variant_id'] !== $variant_Id) {
                                $pIds['variant_id'] = $value['variant_id'];
                            }
                            if ($pIds['product_id'] == $productId && $pIds['variant_id'] == $variant_Id || $pIds['is_sticker_product'] == 1) {
                                $bigAddToCarttData = [
                                    'product_id' => $productId,
                                    'variant_id' => $variant_Id,
                                    'cart_id' => $cart['id'],
                                    'line_item_id' => $cartItemId,
                                    'custom_design_id' => $designID,
                                ];
                                $bigAddToCart = new BigcommerceCart($bigAddToCarttData);
                                $bigAddToCart->save();
                            }
                        }
                    }
                }
                $cartItemsArr[] = $cart['redirect_urls']['cart_url'];
    
            }catch(\Exception $e){
                $cartItemsArr = [
                    'status' => 0,
                    'message' => 'Missing required fields',
                    'exception' => $e->getMessage(),
                ];
            }
            return $cartItemsArr;
        }else{
            throw new \Exception("Some cart value are missing");
        }
        
    }

    /**
     * Multi-sheet add-to-cart-api.
     * 
     * @author sonali@imprintnext.com
     * @date 09 Jun 2023
     * @param cart details
     * @return Cart parameter
     */
    public function createStickerVariant($cartData)
    {   
        $optionsData = $this->call('GET', 'catalog/products/'.$cartData['product_id'].'/options');
        $optionsData = json_decode($optionsData,true);
        $optionsData = $optionsData['data'];
        $optionArr = array_column($optionsData, 'display_name');
        if(!in_array("sheets", $optionArr)){
          
             $optionsArr = [
                "product_id" => $cartData['product_id'],
                "name" => "Sheets",
                "is_visible" => false,
                "display_name" => "sheets",
                "type" => "dropdown",
                "option_values" => [
                    [
                        "is_default" => false,
                        "label" => $cartData['sheet_name']
                    ]
                ]
            ];
           $getOptionData = $this->call("POST", "catalog/products/".$cartData['product_id']."/options",$optionsArr);
           $getOptionData = json_decode($getOptionData, true);
           $allOptionValues[] = [
                "id" => $getOptionData['data']['option_values'][0]['id'],
                "option_id" => $getOptionData['data']['id']
            ];
        }else{
            foreach($optionsData as $options){
                if($options['display_name'] == "sheets"){
                    $optionValues = [
                        "label" => $cartData['sheet_name'],
                        "sort_order" => 0
                    ]; 
                    $getOptionData = $this->call("POST", "catalog/products/".$cartData['product_id']."/options/".$options['id']."/values",$optionValues);
                    $getOptionData = json_decode($getOptionData, true);
                    $allOptionValues[] = [
                        "id" => $getOptionData['data']['id'],
                        "option_id" => $options['id']
                    ];
                }
            }
        }
        $productDetails = $this->call("GET", "catalog/products/".$cartData['product_id']."/variants/".$cartData['variant_id']);
        $productDetails = json_decode($productDetails, true);
        if(!empty($productDetails['data']['option_values'])){
            $allOptionValues[] = [
                "id" => $productDetails['data']['option_values'][0]['id'],
                "option_id" => $productDetails['data']['option_values'][0]['option_id']
            ];

        }
        $getProductImage = $productDetails = $this->call("GET", "catalog/products/".$cartData['product_id']."/variants");
        $getProductImage = json_decode($getProductImage, true);
        $productImage = !empty($productDetails['data']['image_url']) ? $productDetails['data']['image_url'] : $getProductImage['data'][0]['image_url'];
        $variantParam = array(
            "price" => $cartData['added_price'],
            "inventory_level" => '99',
            "image_url" => $productImage,
            "sku" => !empty($productDetails['data'][0]['sku_id']) ? $productDetails['data'][0]['sku_id'].'_'.rand(500,7000) : "big".'_'.rand(500,7000),
            "option_values" => $allOptionValues
        );
        $createVariant = $this->call("POST", "catalog/products/".$cartData['product_id']."/variants", $variantParam);
        $createVariant = json_decode($createVariant,true);
        $createVariant = $createVariant['data'];
        return [
            'list_price' => $createVariant['price'],
            'quantity' => $cartData['qty'],
            'product_id' => $createVariant['product_id'],
            'variant_id' => $createVariant['id'],
        ];
    }

    /*     * ***************************** Cart Module End ***************************** */

     /**
     * Internal function to get sanitized API text contents
     *
     * @return string
     */
    public static function bigCommerce_body($body)
    {
        return strip_tags(str_ireplace('"', "''", $body));
    }

    /**
     * Bigcommerce REST API call helper function
     *
     * @param $filters Product Name & variant Name 
     *
     * @author steve@imprintnext.com
     * @date   15 jan 2022
     * @return return status product exists
     */
    public function SkuNameValidate($filters, $isCustomize){
        $productId = 0; 
        $sku = $filters['sku'];
        $productName = $filters['name'];
        if(isset($sku) && $sku !=''){
            $productParams['sku'] = urlencode($sku);
            $product = $this->call('GET', 'catalog/products',$productParams);
            $product = json_clean_decode($product, true);
            $product = $product['data'];
            if(!empty($product)) {
                $productId = $product[0]['id']; 
            }
        }
        if(isset($productName) && $productName !=''){
            $productParams['name'] = urlencode($productName);
            $product = $this->call('GET', 'catalog/products',$productParams);
            $product = json_clean_decode($product, true);
            $product = $product['data']; 
            if(!empty($product)) {
                $productId = $product[0]['id']; 
            }
        }
        return $productId;
    }

    /**
     * Bigcommerce REST API call helper function
     *
     * @param 
     *
     * @author 
     * @date   15 jan 2022
     * @return return status product exists
     */
    public function orderStatus(){
         $status = [];
         $dataDetails = [];
         $getStatus = $this->call('GET', 'order_statuses',$status,'v2');
         $getStatusArr = json_decode($getStatus,true);
         foreach ($getStatusArr as $statusValue) {
            $value = array(
                "key" => $statusValue['id'],
                "value" => $statusValue['name']
                
            );
            $dataDetails[] = $value;
        }
        
        return $dataDetails;
    }

    /**
     * Bigcommerce REST API call helper function
     *
     * @param $productId Product ID 
     *
     * @author steve@imprintnext.com
     * @date   15 jan 2022
     * @return return product mnetadata
     */
    public function getMetafieldDetails($productId){
        $design_reff_id = 0; 
        $status = [];
        $metaDetails = $this->call('GET', 'catalog/products/'.$productId.'/metafields',$status,'v3');
        $metaDetails = json_decode($metaDetails,true);
        $metaDetails = $metaDetails['data'];
        if(!empty($metaDetails)){
            foreach ($metaDetails as  $value) {
                if($value['key']=='imprint_design_id'){
                    $design_reff_id = $value['value'];
                    $design_reff_id = explode("_",$design_reff_id);
                    $design_reff_id = $design_reff_id[1];
                }
            }
        }
        return $design_reff_id;
    }

    /**
     * Bigcommerce REST API call helper function
     *
     * @param $args cart_id and line_item_id  
     *
     * @author steve@imprintnext.com
     * @date   15 jan 2022
     * @return int Custom design Id
     */
    public function storeAddToCart($args){
        $designReffId = 0;       
        $cartId = $args['cartid'];
        $cart_line_item = $args['cart_line_item'];
        $status = [];
        $cartDetails = $this->call('GET', 'carts/'.$cartId,$status,'v3');
        $cartDetails = json_decode($cartDetails,true);
        $cartDetails = $cartDetails['data'];
        $physical_items = $cartDetails['line_items']['physical_items'];
        foreach ($physical_items as $value) {
            if($value['id'] == $cart_line_item ){
                $productId = $value['product_id'];
                $variantId = $value['variant_id'];
                $designReffId = $this->getMetafieldDetails($productId);
                if($designReffId > 0){
                    $bigAddToCarttData = [
                        'product_id' => $productId,
                        'variant_id' => $variantId,
                        'cart_id' => $cartId,
                        'line_item_id' => $cart_line_item,
                        'custom_design_id' => $designReffId,
                    ];
                    $bigAddToCart = new BigcommerceCart($bigAddToCarttData);
                    $sql = $bigAddToCart->where("cart_id" ,'=',$cartId)
                    ->where("line_item_id" ,'=',$cart_line_item)
                    ->select('*')->get()->toArray();                    
                    if($sql == []){
                        $bigAddToCart->save();                      
                    }                 
                }
            }
        }
        return $designReffId; 
    }

    /**
     * Get: variant id 
     *
     * @param $args cart_id and line_item_id  
     *
     * @author divya@imprintnext.com
     * @date   15 jan 2022
     * @return int Custom design Id
     */
    public function getOnlyVariants($productId,$optionValue){
        $allAtt = array();
        $variants = $this->call('GET', 'catalog/products/'.$productId.'/variants');
        $variantArray = json_clean_decode($variants, true); 
        foreach($variantArray['data'] as $variantData)
         {  
                 $allAttrs = array( 
                     "variant_id" => $variantData['id'],
                     "sku" => $variantData['sku']
                 );
                 $allAtt[] = $allAttrs;
             }
             return  $allAtt;
    }

    /**
     * Bigcommerce REST API call helper function
     *
     * @author sonali@imprintnext.com
     * @date   15 jan 2022
     * @return int Custom design Id
     */
    public function countOrders(){
        $ordersCount = $this->call('GET', 'orders/count', [],'v2');
        $ordersCount = json_decode($ordersCount,true);
        return $ordersCount['count'];
     }

     /**
     * Bigcommerce REST API call helper function
     *
     * @author sonali@imprintnext.com
     * @date   15 jan 2022
     * @return int Custom design Id
     */
     public function getProductStock($productId,$variantId){
        $product = $this->call('GET', 'catalog/products/'.$productId);
        $product = json_decode($product,true);
        $product = $product['data'];
        $stuckDetails = $this->call('GET', 'catalog/products/'.$productId.'/variants/'.$variantId);
        $stuckDetails = json_decode($stuckDetails, true);
        return [
            'productname' => $product['name'],
            'productsku' => $stuckDetails['data']['sku'],
            'stock' => $stuckDetails['data']['inventory_level'],
        ];
     }

    /**
     * Bigcommerce REST API call helper function
     *
     * @param $categoryName  
     *
     * @author steve@imprintnext.com
     * @date   15 jan 2022
     * @return category Id
     */
    private function createCollection($categoryName){
        $msg = '';
        $status = 0;
        $colID = $this->checkCreateCollection($categoryName);
        if ($colID == 0) {                  
            try {
                $collectionArray = array(
                    "parent_id" => 0,
                    "name" => $categoryName,
                    "description" => "The products under this collection are allowed to be shown in designer tool",
                    "sort_order" => 3,
                    "meta_keywords" => [],
                    "layout_file" => "category.html",
                    "default_product_sort" => "use_store_settings",
                    "custom_url" => ["url" => "/".$categoryName."/","is_customized" => false],
                    "views" => 1050,
                );
                $createCollection = $this->call('POST', 'catalog/categories', $collectionArray,'v3');
                $createCollection = json_decode($createCollection,true);
                $createCollection = $createCollection['data'];
                if(!empty($createCollection)){  
                    $status = 1;
                    $colID = $createCollection['id'];
                    $msg = $categoryName . " Category has been created.";
                    }
                
            } catch (\Exception $e){
                $msg = 'Category creation error: ' . $e->getMessage();
                $this->xe_log("\n" . date("Y-m-d H:i:s") . ': Error : ' . $msg . ' : ' . $e->getMessage() . "\n");
            }
        }
        else {
            $msg = $categoryName . " Category Already available.";
            $status = 1;
        }
        return array($status,$msg,$colID);
    }


    /**
     * Bigcommerce REST API call helper function
     *
     * @param $categoryName  
     *
     * @author steve@imprintnext.com
     * @date   15 jan 2022
     * @return category Id
     */
    public function checkCreateCollection($categoryName){    
        $thisColID = 0;
        $params = ['name' => $categoryName];
        $response = $this->call('GET', 'catalog/categories',$params);
        try {
            $data = json_decode($response,true);
            if(!empty($data['data']) ){
                $thisColID = $data['data'][0]['id'];
            }
        } catch (\Exception $e) {
            $msg = 'Category creation error: ' . $e->getMessage();
            $this->xe_log("\n" . date("Y-m-d H:i:s") . ': Error : ' . $msg . ' : ' . $e->getMessage() . "\n");
        }  
        return $thisColID;
    }

    /**
     * Bigcommerce REST API call helper function
     *
     * @param $categoryName  
     *
     * @author steve@imprintnext.com
     * @date   16 Mar 2022
     * @return Array Of variants
     */
    public function getStoreVariants($productId)
    {
        $variants = [];
        $response = $this->call('GET', 'catalog/products/' . $productId . '/variants');
        $response = json_clean_decode($response, true);
        if (!empty($response['data'])) {
            $variantArray = $response['data'];
            foreach ($variantArray as $value) {
                $sizeName = '';
                $colorName = '';
                foreach ($value['option_values'] as $optionValue) {
                    if (strtolower($optionValue['option_display_name']) == 'size') {
                        $sizeName = $optionValue['label'];
                    }
                    if (strtolower($optionValue['option_display_name']) == 'color') {
                        $colorName = $optionValue['label'];
                    }
                }
                $variants[] = [
                    "id" => $value['id'],
                    "title" => $colorName . " / " . $sizeName,
                    "size" => $sizeName,
                    "color" => $colorName,
                    "price" => $value['calculated_price'],
                    "sku" => $value['sku'],
                    "inventory" => $value['inventory_level']
                ];
            }
        }
        return $variants;
    }

    /**
     * Will give all avialable attribute combination
     *
     * @param $arrays, $i - loop increment  
     *
     * @author divya@imprintnext.com
     * @date   15 Apr 2022
     * @return Array Of variants
     */
    private function variantCombinations($arrays, $i = 0) {
        if (!isset($arrays[$i])) {
            return array();
        }
        if ($i == count($arrays) - 1) {
            return $arrays[$i];
        }
        // get combinations from subsequent arrays
        $tmp = $this->variantCombinations($arrays, $i + 1);
        $combinations = array();
        // concat each array from tmp with each element from $arrays[$i]
        foreach ($arrays[$i] as $v) {
            foreach ($tmp as $t) {
                $combinations[] = is_array($t) ? 
                array_merge(array($v), $t) :
                array($v, $t);
            } 
        }
        return $combinations;
    }

    /**
     * Variant id of a product
     *
     * @param $productId  
     *
     * @author divya@imprintnext.com
     * @date   14 June 2022
     * @return Variant id (intiger)
     */
    public function getVariantId($productId){
        $productData = $this->call('GET', 'catalog/products/'.$productId.'/variants');
        $variantId = json_clean_decode($productData, true);
        return $variantId['data'][0]['id'];
    }


    /**
     * Get Order w.r.t Customer
     *
     * @author     divya@imprintnext.com
     * @date       31 Oct 2022
     * @param  customer id
     * @response   Array of order details
     */
    public function getOrderDetailsByCustomer($customerID) {
        $customParam['customer_id']= $customerID;
        $customParam['include'] = 'consignments.line_items';
        $customerOrder = [];
        if ($customerID != '') {
            $customerOrder = $this->call('GET', 'orders',$customParam,'v2');
        }
        return $customerOrder;
    }
    /**
     * Get Order short info for purchase order
     *
     * @author     sonali@imprintnext.com
     * @date       08 nov 2022
     * @param  customer id
     * @response   Array of order details
     */

    public function getOrderShotInfo($orderIds,$storeId){
        foreach($orderIds as $orderId){
            $ordersData = [];
            $orderDetails = $this->call('GET', 'orders/'.$orderId ,$ordersData,'v2');
            $orderDetails = json_decode($orderDetails,true);
            $orderData[] = [
                "id" => $orderDetails['id'],
                "order_number" => $orderDetails['id'],
                "order_total_quantity" => $orderDetails['items_total'],
                "customer_first_name" => $orderDetails['billing_address']['first_name'],
                "customer_last_name" => $orderDetails['billing_address']['last_name'],
                "created_date" => $orderDetails['date_created'],
                "status" => $orderDetails['status'],
            ];
          
        }
        return $orderData;
    }

    public function variantStockUpdate($productStockArr = [])
    {
        $status = 0;
        foreach ($productStockArr as $selectedValue) {
            $selectedProId = $selectedValue['productId'];
            $selectedVarId = $selectedValue['variantId'];
            $stock = $selectedValue['stock'];
            $price = $selectedValue['price'];
            $status = 1;
            $variantArray = array(
                "price" => $price,
                "inventory_level" => $stock,
                "inventory_warning_level" => $stock,
                // "id" => $selectedVarId,
                "product_id" => $selectedProId,
            );
            $updateVariant = $this->call('PUT', 'catalog/products/' . $selectedProId . '/variants/' . $selectedVarId, $variantArray, 'v3');
            if (!empty($updateVariant)) {
                $status = 1;
            }
        }
        return [
            'status' => $status
        ];
    }

    /**
     * Create Reorder
     *
     * @author     steve@imprintnext.com
     * @date       21 Apr 2023
     * @param  orderId
     * @param  order item no.
     * @response   checkout url
     */
    public function createStoreReorder($orderId, $orderItemIds)
    {
        $orderItemIds = ltrim($orderItemIds,"#");
        $orderItemIds = explode("#",$orderItemIds);        

        $orderParams = [];
        $orders = $this->call('GET', 'orders/'.$orderId ,$ordersData,'v2');
        $ordersArr = json_decode($orders,true);
        $getItems = $this->call('GET', 'orders/'. $orderId.'/products',$orderParams ,'v2');
        $getItemArr = json_decode($getItems,true);
        foreach($getItemArr as $cartKey=> $itemValue){
            $productId = $itemValue['product_id'];
            $varientId = $itemValue['variant_id'];
            $itemId = $itemValue['id'];
            if (in_array($itemId,$orderItemIds)){
                $cartId = $ordersArr['cart_id'];
                if(!empty($cartId)){
                    $bigCart = new BigcommerceCart();
                    $sql = $bigCart->where("cart_id" ,'=',$cartId)
                    ->where("product_id" ,'=',$productId)
                    ->where("variant_id" ,'=',$varientId)
                ->select('custom_design_id')->get()->toArray();
                
                    if($sql == []){
                        $refID = 0;
                    }else{
                        $refID = $sql[0]['custom_design_id'];       
                    }
                }else{
                    $bigCart = new BigcommerceCart();
                    $sql = $bigCart->where(
                    "order_id" ,'=',$orderId)
                    ->where("product_id" ,'=',$productId)
                    ->where("variant_id" ,'=',$varientId)
                ->select('custom_design_id')->get()->toArray();
                    if($sql == []){
                        $refID = 0;
                    }else{
                        $refID = $sql[0]['custom_design_id'];       
                    }
                }
            
                $key =[];
                foreach ($itemValue['product_options'] as $optionValue) {
                    if($optionValue['display_name'] == 'Size'){
                        $sizename = $optionValue['display_value_customer'];
                    }
                    if($optionValue['display_name'] == 'Color'){
                        $colorName = $optionValue['display_value_customer'];
                    }
                    if(strtolower($optionValue['display_name']) == 'stitch_color'){
                        $stitchColorName = $optionValue['display_value_customer'];
                    }
                    $keyName = $optionValue['display_value_customer'];
                    $key[]= $keyName;  
                }
                $productVariant = implode(" , ",$key);
                $item = [
                    "id" => $itemValue['id'],
                    "product_id" => $productId,
                    "variant_id" => $varientId,
                    "name" => (!empty($itemValue['product_options'])) ? $itemValue['name'].' - '.$productVariant : $itemValue['name'],
                    "price" => (float)round($itemValue['base_price'],2),
                    "quantity" => $itemValue['quantity'],
                    "total" => (float)round(($itemValue['base_price'] * $itemValue['quantity']),2),
                    "sku" => $itemValue['sku'],
                    "custom_design_id" => $refID,
                    "size" =>  $sizename,
                    "color" => $colorName,
                    "stitch_color" => $stitchColorName,
                    'productDecorationSettingData' => '',
                    'weight_unit' => 'kg',
                    'weight' => $itemValue['weight'],
                ];
                $lineItems[] =$item;           


                $cartParams[$cartKey]['product_id'] = $productId;
                $cartParams[$cartKey]['variant_id'] = $varientId;
                $cartParams[$cartKey]['quantity'] = $itemValue['quantity'];            
                $cartParams[$cartKey]['list_price'] = $itemValue['base_price'];
                $cartParams['include'] = 'redirect_urls';
                $cartParams['module'] = 'cart';           
                
               //End cart
            }
        }
        $path = 'carts';
            $cart = $this->call('POST', $path, $cartParams, 'v3');
            $cart = json_clean_decode($cart, true);
            $cart = $cart['data'];
            $line_items = $cart['line_items'];
            $physical_items = $line_items['physical_items'];

            foreach ($physical_items as $key => $value) {
                $cartItemId =  $value['id'];
                $cartProductId =  $value['product_id'];
                $cartVariantId =  $value['variant_id'];

                foreach($lineItems as $lineItem){        
                    $lineItemProductId =  $lineItem['product_id'];
                    $lineItemVariantId =  $lineItem['variant_id'];

                    if($lineItemProductId == $cartProductId && $lineItemVariantId == $cartVariantId and $lineItem['custom_design_id'] > 0){
                        $bigAddToCarttData = [
                            'product_id' => $cartProductId,
                            'variant_id' => $cartVariantId,
                            'cart_id' => $cart['id'],
                            'line_item_id' => $cartItemId,
                            'custom_design_id' => $lineItem['custom_design_id'],
                        ];
                        $bigAddToCart = new BigcommerceCart($bigAddToCarttData);
                         $bigAddToCart->save();                      
                    }                    
                }
            }
        return($cartItemsArr[] = $cart['redirect_urls']['cart_url']);
    }
}